package report_form

import (
	"errors"
	"fmt"
	"strings"
	"time"
	"unicode/utf8"
)

type Table struct {
	name            string
	keyList         []string
	maxLenValueList []int
	recordList      [][]string
}

func NewTable(name string, keys ...string) *Table {
	maxLenValueList := make([]int, len(keys))
	for i, key := range keys {
		maxLenValueList[i] = strFormatLen(key)
	}

	return &Table{
		name:            name,
		keyList:         keys,
		recordList:      make([][]string, 0),
		maxLenValueList: maxLenValueList,
	}
}

func strFormatLen(s string) int {
	l := 0
	for i := 0; i < len(s); {
		_, size := utf8.DecodeRuneInString(s[i:])
		if size != 1 {
			l += 2
		} else {
			l++
		}
		i += size
	}
	return l
}

func (t *Table) ToString() string {
	const spacing = 4
	rtn := new(strings.Builder)

	//标题
	totalLength := spacing * (len(t.maxLenValueList) - 1)
	for _, valueLen := range t.maxLenValueList {
		totalLength += valueLen
	}
	preBlankLen := (totalLength - strFormatLen(t.name)) / 2
	postBlankLen := totalLength - preBlankLen - strFormatLen(t.name)

	for i := 0; i < preBlankLen; i++ {
		rtn.WriteString("-")
	}
	rtn.WriteString(t.name)
	for i := 0; i < postBlankLen; i++ {
		rtn.WriteString("-")
	}
	rtn.WriteString("\n")

	//key
	for i, key := range t.keyList {
		rtn.WriteString(key)

		//补全空格，最后一行结尾不打印分隔空白
		blankLen := t.maxLenValueList[i] - strFormatLen(key) + spacing
		if i == len(t.keyList)-1 {
			blankLen -= spacing
		}
		for i := 0; i < blankLen; i++ {
			rtn.WriteString(" ")
		}
	}
	rtn.WriteString("\n")

	//value
	for _, valueList := range t.recordList {
		for i, value := range valueList {
			rtn.WriteString(value)

			blankLen := t.maxLenValueList[i] - strFormatLen(value) + spacing
			if i == len(t.keyList)-1 {
				blankLen -= spacing
			}
			for i := 0; i < blankLen; i++ {
				rtn.WriteString(" ")
			}
		}
		rtn.WriteString("\n")
	}

	return rtn.String()
}

//添加记录
func (t *Table) AddRecord(values ...interface{}) error {
	if len(values) != len(t.keyList) {
		return errors.New("len(values) != len(keyList)")
	}

	//interface转string，并且记录最大长度
	valueList := make([]string, 0)
	for i, v := range values {
		value := fmt.Sprint(v)
		valueList = append(valueList, value)
		if strFormatLen(value) > t.maxLenValueList[i] {
			t.maxLenValueList[i] = strFormatLen(value)
		}
	}

	t.recordList = append(t.recordList, valueList)
	return nil
}

func GeneralTableRun(sm *ServiceMap) string {
	const (
		cpuMin  = 1
		memMin  = 1
		diskMin = 1
	)

	//生成报表
	rtn := new(strings.Builder)

	//表头
	rtn.WriteString("                        服务健康状态总表\n")

	//时间
	rtn.WriteString(fmt.Sprintf("时间：%s ～ %s\n",
		sm.startTime.Format(time.RFC3339), sm.endTime.Format(time.RFC3339)))

	//访问量
	rtn.WriteString(fmt.Sprintf("总访问量：%v  总响应时间：%v  平均响应时间：%v\n\n",
		sm.GetCount(), sm.GetSumDuration(), sm.GetAverageDuration()))

	//服务平均响应时间排行
	serviceTable := NewTable("服务平均响应时间排行", "duration", "sys_name", "traffic_volume")
	serviceList := sm.GetAverageDurationServiceList()
	for _, s := range serviceList {
		_ = serviceTable.AddRecord(s.GetAverageDuration(), s.Name(), s.GetCount())
	}
	rtn.WriteString(serviceTable.ToString())
	rtn.WriteString("\n")

	//服务响应时间中位值排行
	medianTable := NewTable("服务接口响应时间中位值排行", "duration", "sys_name", "path", "traffic_volume")
	medianList := sm.GetMedianDurationPathList()
	for _, m := range medianList {
		_ = medianTable.AddRecord(m.GetMedianDuration(), m.GetServiceName(), m.GetPath(), m.GetCount())
	}
	rtn.WriteString(medianTable.ToString())
	rtn.WriteString("\n")

	//服务响应时间平均值排行
	averageTable := NewTable("服务接口响应时间平均值排行", "duration", "sys_name", "path", "traffic_volume")
	averageList := sm.GetAverageDurationPathList()
	for _, m := range averageList {
		_ = averageTable.AddRecord(m.GetAverageDuration(), m.GetServiceName(), m.GetPath(), m.GetCount())
	}
	rtn.WriteString(averageTable.ToString())
	rtn.WriteString("\n")

	//响应时间最长的请求排行
	maxTable := NewTable("服务接口响应时间最高值排行", "duration", "sys_name", "path", "timestamp", "trace_id")
	maxList := sm.GetMaxDurationTracePointList()
	for _, m := range maxList {
		_ = maxTable.AddRecord(m.Duration, m.ServiceName, m.Path, m.Timestamp, m.TraceId)
	}
	rtn.WriteString(maxTable.ToString())
	rtn.WriteString("\n")

	//去topN100后的峰值cpu使用率排行
	maxRemoveN100CpuTable := NewTable("常规cpu使用率排行", "cpu", "sys_name", "traffic_volume", "average_duration")
	maxRemoveN100CpuList := sm.GetRemoveN100MaxCpuServiceList()
	for _, s := range maxRemoveN100CpuList {
		if s.GetRemoveN100MaxCpu() < cpuMin {
			break
		}
		_ = maxRemoveN100CpuTable.AddRecord(s.GetRemoveN100MaxCpu(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(maxRemoveN100CpuTable.ToString())
	rtn.WriteString("\n")

	//峰值cpu使用率排行
	maxCpuTable := NewTable("峰值cpu使用率排行", "cpu", "sys_name", "traffic_volume", "average_duration")
	maxCpuList := sm.GetMaxCpuServiceList()
	for _, s := range maxCpuList {
		if s.GetMaxCpu() < cpuMin {
			break
		}
		_ = maxCpuTable.AddRecord(s.GetMaxCpu(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(maxCpuTable.ToString())
	rtn.WriteString("\n")

	//内存峰值使用率排行
	memTable := NewTable("峰值内存使用率排行", "mem", "sys_name", "traffic_volume", "average_duration")
	memList := sm.GetMaxMemServiceList()
	for _, s := range memList {
		if s.GetMaxMem() < memMin {
			break
		}
		_ = memTable.AddRecord(s.GetMaxMem(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(memTable.ToString())
	rtn.WriteString("\n")

	//硬盘峰值使用率排行
	diskTable := NewTable("峰值硬盘使用率排行", "disk", "sys_name", "traffic_volume", "average_duration")
	diskList := sm.GetMaxDiskServiceList()
	for _, s := range diskList {
		if s.GetMaxDisk() < diskMin {
			break
		}
		_ = diskTable.AddRecord(s.GetMaxDisk(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(diskTable.ToString())
	rtn.WriteString("\n")

	return rtn.String()
}

func GeneralTableNewSM(n int) *ServiceMap {
	//求前n天的数据，按天取整
	now := time.Now()
	startTime := now.Add(-1 * time.Duration(now.Hour()) * time.Hour)
	startTime = startTime.Add(-1 * time.Duration(now.Minute()) * time.Minute)
	startTime = startTime.Add(-1 * time.Duration(now.Second()) * time.Second)
	startTime = startTime.Add(-1 * time.Duration(now.Nanosecond()))
	startTime = startTime.AddDate(0, 0, -n)
	endTime := startTime.AddDate(0, 0, n)

	//查询
	sm := NewServiceMap(startTime, endTime)

	return sm
}
