package report_form

import (
	"errors"
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"math"
	"strings"
	"time"
)

const (
	timeFormat = "2006-01-02T15:04:05"
	colorRed   = "red"
	colorBlack = "black"
	cpuMin     = 1
	memMin     = 1
	diskMin    = 1
)

var (
	cstZone     = time.FixedZone("CST", 8*3600)
	maxDuration = time.Millisecond * 600
	topN        = 20
)

//Human-readable number
func hrn(n int) string {
	num := float64(n)
	unit := float64(1024)
	if num < unit {
		return fmt.Sprintf("%d", int(num))
	}
	exp := int(math.Log(num) / math.Log(unit))
	pre := "kMGTPE"
	pre = pre[exp-1 : exp]
	r := float64(n) / math.Pow(unit, float64(exp))
	return fmt.Sprintf("%.3f%s", r, pre)
}

//TODO 分表
func (sm *ServiceMap) ServiceReportForm() map[string]string {
	rtn := make(map[string]string)

	//

	return rtn
}

//获取指定path的访问量
func (sm *ServiceMap) GetPathCount(serviceName string, path string) int {
	s, ok := sm.serviceMap[serviceName]
	if !ok {
		return 0
	}

	pm := s.GetPathMap()
	p, ok := pm[path]
	if !ok {
		return 0
	}

	return p.GetCount()
}

//分表
func (s *Service) ReportForm(long string) string {
	rtn := new(strings.Builder)
	var t *Table = nil

	rtn.WriteString("<html><head><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\" /></head><body><div><pre>\n")

	//表头
	rtn.WriteString("                         服务健康状态表:" + s.Name() + "\n")
	rtn.WriteString(fmt.Sprintf("时间：%s ~ %s\n",
		s.startTime.In(cstZone).Format(timeFormat),
		s.endTime.In(cstZone).Format(timeFormat)),
	)
	rtn.WriteString(fmt.Sprintf("总访问量：%s  总响应时间：%s  平均响应时间：%s  平均QPS：%.2f  MAXQPS：%s\n",
		hrn(s.GetCount()),
		s.GetSumDuration()/1e6*1e6,     //精度保留到毫秒
		s.GetAverageDuration()/1e6*1e6, //精度保留到毫秒
		s.GetQps(),
		hrn(s.GetMaxQps())),
	)
	rtn.WriteString(fmt.Sprintf("常规CPU使用率：%d%%  峰值CPU使用率：%d%%  平均内存使用率：%d%%  峰值内存使用率：%d%%  峰值硬盘使用率：%d%%\n",
		s.GetRemoveN100MaxCpu(),
		s.GetMaxCpu(),
		s.GetAverageMem(),
		s.GetMaxMem(),
		s.GetMaxDisk(),
	))
	url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s\n", global.LocalIp, global.HttpPort,
		long, s.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
	rtn.WriteString(htmlStringA(url, url))
	rtn.WriteString("\n\n")

	//说名
	rtn.WriteString("阅读说明:\n")
	rtn.WriteString("1、易读数字：单位K、M、G、T、P、E，进制1024\n")
	rtn.WriteString("2、排名：排名越高健康状态越差，如果排名标红，则表示存在风险\n")
	rtn.WriteString("3、表头部分的CPU、内存、硬盘使用率：取的是服务节点中各项使用率的的最高值\n")
	rtn.WriteString("4、常规cpu使用率：去针尖后的最高CPU使用率，针尖指的是CPU使用率暴增的场景\n")
	rtn.WriteString("\n\n")

	//服务接口中位响应时间TOP N
	getMedianDurationPathList := s.GetMedianDurationPathList()
	if len(getMedianDurationPathList) != 0 {
		t = NewTable(fmt.Sprintf("服务接口中位响应时间TOP%d", topN), "排名", "响应时间", "接口", "访问量", "MAXQPS")
		for i, p := range getMedianDurationPathList {
			color := colorBlack
			if p.GetMedianDuration() > maxDuration {
				color = colorRed
			}
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", p.GetMedianDuration()/1e6*1e6), //精确到微秒
				p.GetPath(),
				hrn(p.GetCount()),
				hrn(p.GetMaxQps()),
			)
			if i+1 == topN {
				break
			}
		}
		rtn.WriteString(t.ToHtml())
		rtn.WriteString("\n\n")
	}

	//服务接口平均响应时间TOP N
	getAverageDurationPathList := s.GetAverageDurationPathList()
	if len(getAverageDurationPathList) != 0 {
		t = NewTable(fmt.Sprintf("服务接口平均响应时间TOP%d", topN), "排名", "响应时间", "接口", "访问量", "MAXQPS")
		for i, p := range getAverageDurationPathList {
			color := colorBlack
			if p.GetAverageDuration() > maxDuration {
				color = colorRed
			}
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", p.GetAverageDuration()/1e6*1e6), //精确到微秒
				p.GetPath(),
				hrn(p.GetCount()),
				hrn(p.GetMaxQps()),
			)
			if i+1 == topN {
				break
			}
		}
		rtn.WriteString(t.ToHtml())
		rtn.WriteString("\n\n")
	}

	//服务接口峰值响应时间TOP N
	getMaxDurationPathList := s.GetMaxDurationPathList()
	if len(getMaxDurationPathList) != 0 {
		t = NewTable(fmt.Sprintf("服务接口峰值响应时间TOP%d", topN), "排名", "响应时间", "接口", "时间戳", "trace_id", "访问量")
		for i, p := range getMaxDurationPathList {
			tp := p.GetMaxDurationTracePoint()
			color := colorBlack
			if tp.Duration > maxDuration {
				color = colorRed
			}
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", tp.Duration/1e6*1e6),
				tp.Path,
				tp.Timestamp.In(cstZone).Format(timeFormat),
				fmt.Sprintf(`<a href="http://zipkin-3c.xyqb.com/zipkin/traces/%s" target="_blank" rel="noopener noreferrer">%s</a>`, tp.TraceId, tp.TraceId),
				hrn(p.GetCount()),
			)
			if i+1 == topN {
				break
			}
		}
		rtn.WriteString(t.ToHtml())
		rtn.WriteString("\n\n")
	}

	//节点状态
	nodeList := s.GetNodeList()
	if len(nodeList) != 0 {
		t = NewTable("节点状态", "IP", "常规CPU使用率", "峰值CPU使用率", "平均内存使用率", "峰值内存使用率", "峰值硬盘使用率", "最大线程数", "平均线程数")
		for _, n := range nodeList {
			removeN100Color, maxCpuColor, averageMemColor, maxMemColor, maxDiskColor := colorBlack, colorBlack, colorBlack, colorBlack, colorBlack
			if n.GetRemoveN100MaxCpu() > 50 {
				removeN100Color = colorRed
			}
			if n.GetMaxCpu() > 80 {
				maxCpuColor = colorRed
			}
			if n.GetAverageMem() > 50 {
				averageMemColor = colorRed
			}
			if n.GetMaxMem() > 80 {
				maxMemColor = colorRed
			}
			if n.GetMaxDisk() > 80 {
				maxDiskColor = colorRed
			}

			//htmlStringColor
			_ = t.AddRecord(
				colorBlack,
				n.GetAddress(),
				htmlStringColor(removeN100Color, fmt.Sprintf("%d%%", n.GetRemoveN100MaxCpu())),
				htmlStringColor(maxCpuColor, fmt.Sprintf("%d%%", n.GetMaxCpu())),
				htmlStringColor(averageMemColor, fmt.Sprintf("%d%%", n.GetAverageMem())),
				htmlStringColor(maxMemColor, fmt.Sprintf("%d%%", n.GetMaxMem())),
				htmlStringColor(maxDiskColor, fmt.Sprintf("%d%%", n.GetMaxDisk())),
				hrn(n.GetMaxThread()),
				hrn(n.GetAverageThread()),
			)
		}
		rtn.WriteString(t.ToHtml())
		rtn.WriteString("\n\n")
	}

	rtn.WriteString("</pre></div></body></html>\n")
	return rtn.String()
}

//html字符串显示颜色
func htmlStringColor(color string, s string) string {
	return fmt.Sprintf(`<font color="%s">%s</font>`, color, s)
}

//链接
func htmlStringA(url string, s string) string {
	return fmt.Sprintf(`<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>`, url, s)
}

//总表
func (sm *ServiceMap) ReportForm(long string) string {
	rtn := new(strings.Builder)
	var t *Table = nil

	rtn.WriteString("<html><head><meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\" /></head><body><div><pre>\n")

	//表头
	rtn.WriteString("                         服务健康状态总表\n")
	rtn.WriteString(fmt.Sprintf("时间：%s ~ %s\n",
		sm.startTime.In(cstZone).Format(timeFormat),
		sm.endTime.In(cstZone).Format(timeFormat)),
	)
	rtn.WriteString(fmt.Sprintf("总访问量：%s  总响应时间：%s  平均响应时间：%s  平均QPS：%.2f  MAXQPS：%s\n",
		hrn(sm.GetCount()),
		sm.GetSumDuration()/1e6*1e6,     //精度保留到毫秒
		sm.GetAverageDuration()/1e6*1e6, //精度保留到毫秒
		sm.GetQps(),
		hrn(sm.GetMaxQps()),
	))
	url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s\n", global.LocalIp, global.HttpPort,
		long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), "all")
	rtn.WriteString(htmlStringA(url, url))
	rtn.WriteString("\n")

	//说名
	rtn.WriteString("阅读说明:\n")
	rtn.WriteString("1、易读数字：单位K、M、G、T、P、E，进制1024\n")
	rtn.WriteString("2、排名：排名越高健康状态越差，如果排名标红，则表示存在风险\n")
	rtn.WriteString("3、CPU、内存、磁盘使用率：取的是服务节点中各项使用率的的最高值\n")
	rtn.WriteString("4、常规cpu使用率：去针尖后的最高CPU使用率，针尖指的是CPU使用率暴增的场景\n")
	rtn.WriteString("\n\n")

	//服务平均响应时间排行
	averageDurationServiceList := sm.GetAverageDurationServiceList()
	if len(averageDurationServiceList) != 0 {
		t = NewTable("服务平均响应时间排行", "排名", "响应时间", "服务名称", "访问量", "MAXQPS")
		for i, s := range averageDurationServiceList {
			color := colorBlack
			if s.GetAverageDuration() > maxDuration {
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6), //精度毫秒
				htmlStringA(url, s.Name()),
				hrn(s.GetCount()),
				hrn(s.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//服务接口响应时间中位值排行
	getMedianDurationPathList := sm.GetMedianDurationPathList()
	if len(getMedianDurationPathList) != 0 {
		t = NewTable("服务接口中位响应时间排行", "排名", "响应时间", "服务名称", "接口", "访问量", "MAXQPS")
		for i, p := range getMedianDurationPathList {
			color := colorBlack
			if p.GetMedianDuration() > maxDuration {
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), p.GetServiceName())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", p.GetMedianDuration()/1e6*1e6), //精度保留毫秒
				htmlStringA(url, p.GetServiceName()),
				p.GetPath(),
				hrn(p.GetCount()),
				hrn(p.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//服务接口响应时间平均值排行
	getAverageDurationPathList := sm.GetAverageDurationPathList()
	if len(getAverageDurationPathList) != 0 {
		t = NewTable("服务接口平均响应时间排行", "排名", "响应时间", "服务名称", "接口", "访问量", "MAXQPS")
		for i, p := range getAverageDurationPathList {
			color := colorBlack
			if p.GetAverageDuration() > maxDuration {
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), p.GetServiceName())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", p.GetAverageDuration()/1e6*1e6), //精度保留毫秒
				htmlStringA(url, p.GetServiceName()),
				p.GetPath(),
				hrn(p.GetCount()),
				hrn(p.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//响应时间最高的请求排行
	getMaxDurationTracePointList := sm.GetMaxDurationTracePointList()
	if len(getMaxDurationTracePointList) != 0 {
		t = NewTable("响应时间最高的请求排行", "排名", "响应时间", "服务名称", "接口", "时间戳", "trace_id", "访问量")
		for i, tp := range getMaxDurationTracePointList {
			color := colorBlack
			if tp.Duration > maxDuration {
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), tp.ServiceName)
			//访问量
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%v", tp.Duration/1e6*1e6), //精度保留毫秒
				htmlStringA(url, tp.ServiceName),
				tp.Path,
				tp.Timestamp.In(cstZone).Format(timeFormat),
				htmlStringA("http://zipkin-3c.xyqb.com/zipkin/traces/"+tp.TraceId, tp.TraceId),
				hrn(sm.GetPathCount(tp.ServiceName, tp.Path)),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//常规cpu使用率排行
	getRemoveN100MaxCpuServiceList := sm.GetRemoveN100MaxCpuServiceList()
	if len(getRemoveN100MaxCpuServiceList) != 0 {
		t = NewTable("常规cpu使用率排行", "排名", "cpu使用率", "服务名称", "访问量", "平均响应时间", "MAXQPS")
		for i, s := range getRemoveN100MaxCpuServiceList {
			if s.GetRemoveN100MaxCpu() < cpuMin {
				break
			}
			color := colorBlack
			if s.GetRemoveN100MaxCpu() > 50 { //超过一半标红
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%d%%", s.GetRemoveN100MaxCpu()),
				htmlStringA(url, s.Name()),
				hrn(s.GetCount()),
				fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
				hrn(s.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//峰值cpu使用率排行
	getMaxCpuServiceList := sm.GetMaxCpuServiceList()
	if len(getMaxCpuServiceList) != 0 {
		t = NewTable("峰值cpu使用率排行", "排名", "cpu使用率", "服务名称", "访问量", "平均响应时间", "MAXQPS")
		for i, s := range getMaxCpuServiceList {
			if s.GetMaxCpu() < cpuMin {
				break
			}
			color := colorBlack
			if s.GetMaxCpu() > 80 { //超过80标红
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%d%%", s.GetMaxCpu()),
				htmlStringA(url, s.Name()),
				hrn(s.GetCount()),
				fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
				hrn(s.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//平均内存使用率排行
	getAverageMemServiceList := sm.GetAverageMemServiceList()
	if len(getAverageMemServiceList) != 0 {
		t = NewTable("平均内存使用率排行", "排名", "内存使用率", "服务名称", "访问量", "平均响应时间", "MAXQPS")
		for i, s := range getAverageMemServiceList {
			if s.GetAverageMem() < memMin {
				break
			}
			color := colorBlack
			if s.GetAverageMem() > 50 { //超过50标红
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%d%%", s.GetAverageMem()),
				htmlStringA(url, s.Name()),
				hrn(s.GetCount()),
				fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
				hrn(s.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//峰值内存使用率排行
	getMaxMemServiceList := sm.GetMaxMemServiceList()
	if len(getMaxMemServiceList) != 0 {
		t = NewTable("峰值内存使用率排行", "排名", "内存使用率", "服务名称", "访问量", "平均响应时间", "MAXQPS")
		for i, s := range getMaxMemServiceList {
			if s.GetMaxMem() < memMin {
				break
			}
			color := colorBlack
			if s.GetMaxMem() > 80 { //超过80标红
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%d%%", s.GetMaxMem()),
				htmlStringA(url, s.Name()),
				hrn(s.GetCount()),
				fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
				hrn(s.GetMaxQps()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	//峰值磁盘使用率排行
	getMaxDiskServiceList := sm.GetMaxDiskServiceList()
	if len(getMaxDiskServiceList) != 0 {
		t = NewTable("峰值硬盘使用率排行", "排名", "硬盘使用率", "服务名称")
		for i, s := range getMaxDiskServiceList {
			if s.GetMaxDisk() < diskMin {
				break
			}
			color := colorBlack
			if s.GetMaxDisk() > 80 {
				color = colorRed
			}
			url := fmt.Sprintf("http://%s:%d/show/%s/%s/%s", global.LocalIp, global.HttpPort,
				long, sm.endTime.Add(time.Hour * -24).In(cstZone).Format("2006/01/02"), s.Name())
			_ = t.AddRecord(
				color,
				fmt.Sprintf("No.%d", i+1),
				fmt.Sprintf("%d%%", s.GetMaxDisk()),
				htmlStringA(url, s.Name()),
			)
		}
		if !t.IsEmpty() {
			rtn.WriteString(t.ToHtml())
			rtn.WriteString("\n\n")
		}
	}

	rtn.WriteString("</pre></div></body></html>\n")

	return rtn.String()
}

type Line struct {
	color  string
	values []string
}

type Table struct {
	name       string
	keyList    []string
	recordList []Line
}

func NewTable(name string, keys ...string) *Table {
	return &Table{
		name:       name,
		keyList:    keys,
		recordList: make([]Line, 0),
	}
}

func (t *Table) IsEmpty() bool {
	return len(t.recordList) == 0
}

func (t *Table) AddRecord(color string, values ...string) error {
	if len(values) != len(t.keyList) {
		return errors.New("len(values) != len(keyList)")
	}

	t.recordList = append(t.recordList, Line{color: color, values: values})
	return nil
}

func (t *Table) ToHtml() string {
	rtn := new(strings.Builder)
	rtn.WriteString("<table>\n")
	rtn.WriteString("\t<style>table{border-spacing: 20px 0px}</style>\n")
	rtn.WriteString(fmt.Sprintf("\t<caption>%s</caption>\n", t.name))
	rtn.WriteString("\t<tr>")
	for _, k := range t.keyList {
		rtn.WriteString(fmt.Sprintf("<td>%s</td>", k))
	}
	rtn.WriteString("</tr>\n")
	for _, x := range t.recordList {
		rtn.WriteString("\t<tr>")
		for _, y := range x.values {
			rtn.WriteString(fmt.Sprintf("<td><font color=\"%s\">%s</font></td>", x.color, y))
		}
		rtn.WriteString("</tr>\n")
	}
	rtn.WriteString("</table>\n")
	return rtn.String()
}

/*
//body := ServiceTableRun(service)
func SubTableRun(s *Service) string {


	//接口平均响应时间排行
	averageDurationTable := NewTable("接口平均响应时间排行", "duration", "path", "traffic_volume")
	plist = s.GetAverageDurationPathList()
	for _, p := range plist {
		_ = averageDurationTable.AddRecord(p.GetAverageDuration(), p.GetPath(), p.GetCount())
	}
	rtn.WriteString(averageDurationTable.ToString())
	rtn.WriteString("\n")

	//接口峰值响应时间排行
	maxDurationTable := NewTable("接口峰值响应时间排行",
		"duration", "path", "timestamp", "trace_id")
	plist = s.GetAverageDurationPathList()
	for _, p := range plist {
		maxPoint := p.GetMaxDurationTracePoint()
		_ = maxDurationTable.AddRecord(maxPoint.Duration,
			maxPoint.Path, maxPoint.Timestamp, maxPoint.TraceId)
	}
	rtn.WriteString(maxDurationTable.ToString())
	rtn.WriteString("\n")

	return rtn.String()
}
*/
