package report_form

import (
	"encoding/json"
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	"time"
)

type TracePoint struct {
	ServiceName string
	Path        string
	TraceId     string
	Duration    time.Duration
	Timestamp   time.Time
}

type Path struct {
	startTime             time.Time     //开始时间
	endTime               time.Time     //结束时间
	serviceName           string        //服务名称
	path                  string        //路径
	count                 int           //访问次数
	averageDuration       time.Duration //平均响应时间
	sumDuration           time.Duration //总响应时间
	medianDuration        time.Duration //中位响应时间
	maxqps                int           //每秒的访问量
	variance              float64       //标准差
	maxDurationTracePoint TracePoint    //最大响应时间的TracePoint
}

//创建Path对象，会访问数据库
func NewPath(serviceName string, path string, startTime time.Time, endTime time.Time) (*Path, bool) {
	rtn := &Path{
		startTime:   startTime,
		endTime:     endTime,
		serviceName: serviceName,
		path:        path,
	}

	//初始化，备注：顺序存在依赖关系
	rtn.initCount()
	rtn.initSumDuration()
	rtn.initAverageDuration()
	rtn.initMedianDuration()
	rtn.initMaxDurationTracePoint()
	rtn.initQps()
	return rtn, true
}

//初始化qps
func (p *Path) initQps() {
	const sqlMaxQps = "SELECT max(qps) as MAXQPS FROM " +
		"(SELECT count(\"traceId\") AS qps FROM trace_info " +
		"WHERE sys_name = '%s' AND path = '%s' AND time >= %d AND time < %d " +
		"GROUP BY time(1s));"
	sql := fmt.Sprintf(sqlMaxQps, p.GetServiceName(), p.GetPath(), p.startTime.UnixNano(), p.endTime.UnixNano())
	maxqps, ok := queryOneAndToInt(sql)
	if ok {
		p.maxqps = maxqps
	}
}

//初始化访问次数
func (p *Path) initCount() {
	const sqlGetCount = `SELECT count("traceId") AS count FROM trace_info ` +
		`WHERE sys_name = '%s' AND path = '%s' AND time >= %d AND time < %d;`
	sql := fmt.Sprintf(sqlGetCount, p.serviceName, p.path, p.startTime.UnixNano(), p.endTime.UnixNano())
	count, ok := queryOneAndToInt(sql)
	if !ok {
		return
	}
	p.count = count
}

//初始化总响应时间
func (p *Path) initSumDuration() {
	const sqlGetAverageDuration = `SELECT sum("duration") AS average FROM trace_info ` +
		`WHERE sys_name = '%s' AND path = '%s' AND time >= %d AND time < %d;`
	sql := fmt.Sprintf(sqlGetAverageDuration, p.serviceName, p.path, p.startTime.UnixNano(), p.endTime.UnixNano())
	sum, ok := queryOneAndToInt(sql)
	if !ok {
		return
	}

	p.sumDuration = time.Duration(sum * 1e6) //毫秒转纳秒
}

//初始化平均时间
func (p *Path) initAverageDuration() {
	//当没有数据时，直接返回
	if p.count == 0 {
		return
	}

	p.averageDuration = p.sumDuration / time.Duration(p.count)
}

//初始化时间中位响应时间
func (p *Path) initMedianDuration() {
	const sqlGetMedianDuration = `SELECT median("duration") AS media FROM trace_info ` +
		`WHERE sys_name = '%s' AND path = '%s' AND time >= %d AND time < %d;`
	sql := fmt.Sprintf(sqlGetMedianDuration, p.serviceName, p.path, p.startTime.UnixNano(), p.endTime.UnixNano())
	median, ok := queryOneAndToInt(sql)
	if !ok {
		return
	}
	p.medianDuration = time.Duration(median * 1e6)
}

//初始化时间最大响应TracePoint
func (p *Path) initMaxDurationTracePoint() {
	const sqlGetMaxDurationTracePoint = `SELECT top("duration", "traceId", 1) AS max FROM trace_info ` +
		`WHERE sys_name = '%s' AND path = '%s' AND time >= %d AND time < %d;`
	sql := fmt.Sprintf(sqlGetMaxDurationTracePoint, p.serviceName, p.path, p.startTime.UnixNano(), p.endTime.UnixNano())
	//	glog.Debug(sql)
	req, err := query(sql)
	if err != nil {
		glog.Error("init max duration trace point:", err)
		return
	}
	if len(req.Results) != 1 {
		return
	}
	resp := req.Results[0]
	if len(resp.Series) != 1 {
		return
	}
	row := resp.Series[0]
	if len(row.Values) != 1 {
		return
	}
	idxTime, ok1 := getModRowKeyIdx("time", row)
	idxMaxDuration, ok2 := getModRowKeyIdx("max", row)
	idxTraceId, ok3 := getModRowKeyIdx("traceId", row)
	if !ok1 || !ok2 || !ok3 {
		return
	}

	//时间戳转换
	time3339, ok := row.Values[0][idxTime].(string)
	if !ok {
		return
	}
	timestamp, err := time.Parse(time.RFC3339, time3339)
	if err != nil {
		return
	}

	//最大时间
	maxDurationJson, ok := row.Values[0][idxMaxDuration].(json.Number)
	if !ok {
		return
	}
	maxDurationInt64, err := maxDurationJson.Int64()
	duration := time.Duration(maxDurationInt64 * 1e6) //毫秒转纳秒
	if err != nil {
		return
	}

	//traceId
	traceId, ok := row.Values[0][idxTraceId].(string)
	if !ok {
		return
	}

	p.maxDurationTracePoint = TracePoint{
		ServiceName: p.serviceName,
		Path:        p.path,
		TraceId:     traceId,
		Duration:    duration,
		Timestamp:   timestamp,
	}
}

//获取开始时间
func (p *Path) GetStartTime() time.Time {
	return p.startTime
}

//获取结束时间
func (p *Path) GetEndTime() time.Time {
	return p.endTime
}

//获取服务名称
func (p *Path) GetServiceName() string {
	return p.serviceName
}

//获取路径
func (p *Path) GetPath() string {
	return p.path
}

//获取访问次数
func (p *Path) GetCount() int {
	return p.count
}

//获取qps
func (p *Path) GetMaxQps() int {
	return p.maxqps
}

//获取总响应时间
func (p *Path) GetSumDuration() time.Duration {
	return p.sumDuration
}

//获取平均响应时间
func (p *Path) GetAverageDuration() time.Duration {
	return p.averageDuration
}

//获取最大响应时间的TracePoint
func (p *Path) GetMaxDurationTracePoint() TracePoint {
	return p.maxDurationTracePoint
}

//获取中位响应时间的TracePoint
func (p *Path) GetMedianDuration() time.Duration {
	return p.medianDuration
}

//获取标准差
func (p *Path) GetVariance() float64 {
	return p.variance
}
