package report_form

import (
	"errors"
	"fmt"
	"math"
	"strings"
	"time"
)

const (
	timeFormat = "2006-01-02T15:04:05"
	colorRed   = "red"
	colorBlack = "black"
	cpuMin     = 1
	memMin     = 1
	diskMin    = 1
)

var (
	cstZone     = time.FixedZone("CST", 8*3600)
	maxDuration = time.Millisecond * 600
)

//Human-readable number
func hrn(n int) string {
	num := float64(n)
	unit := float64(1024)
	if num < unit {
		return fmt.Sprintf("%d", int(num))
	}
	exp := int(math.Log(num) / math.Log(unit))
	pre := "kMGTPE"
	pre = pre[exp-1 : exp]
	r := float64(n) / math.Pow(unit, float64(exp))
	return fmt.Sprintf("%.3f%s", r, pre)
}

//总表
func ServiceMapReportForm(startTime time.Time, endTime time.Time) string {
	rtn := new(strings.Builder)
	sm := NewServiceMap(startTime, endTime)
	var t *Table = nil

	//表头
	rtn.WriteString("<div><pre>\n")
	rtn.WriteString("                         服务健康状态总表\n")
	rtn.WriteString(fmt.Sprintf("时间：%s ~ %s\n",
		startTime.In(cstZone).Format(time.RFC3339),
		endTime.In(cstZone).Format(time.RFC3339)),
	)
	rtn.WriteString(fmt.Sprintf("总访问量：%s  总响应时间：%s  平均响应时间：%s  QPS：%.2f\n",
		hrn(sm.GetCount()),
		sm.GetSumDuration()/1e6*1e6,     //精度保留到毫秒
		sm.GetAverageDuration()/1e6*1e6, //精度保留到毫秒
		sm.GetQps()),
	)
	rtn.WriteString("\n")
	rtn.WriteString("阅读说明:\n")
	rtn.WriteString("1、易读数字：单位K、M、G、T、P、E，进制1024\n")
	rtn.WriteString("2、排名：排名越高健康状态越差，如果排名标红，则表示存在风险\n")
	rtn.WriteString("3、CPU、内存、磁盘：服务有多个节点，总表中CPU、内存、磁盘取的是服务节点中的最高值\n")
	rtn.WriteString("4、常规cpu使用率：去针尖后的最高CPU使用率，针尖指的是CPU使用率暴增的场景\n")
	rtn.WriteString("\n\n")

	//服务平均响应时间排行
	t = NewTable("服务平均响应时间排行", "排名", "响应时间", "服务名称", "访问量", "QPS")
	for i, s := range sm.GetAverageDurationServiceList() {
		color := colorBlack
		if s.GetAverageDuration() > maxDuration {
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6), //精度毫秒
			s.Name(),
			hrn(s.GetCount()),
			fmt.Sprintf("%.2f", s.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//服务接口响应时间中位值排行
	t = NewTable("服务接口中位响应时间排行", "排名", "响应时间", "服务名称", "接口", "访问量", "QPS")
	for i, p := range sm.GetMedianDurationPathList() {
		color := colorBlack
		if p.GetMedianDuration() > maxDuration {
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%v", p.GetMedianDuration()/1e6*1e6), //精度保留毫秒
			p.GetServiceName(),
			p.GetPath(),
			hrn(p.GetCount()),
			fmt.Sprintf("%.2f", p.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//服务接口响应时间平均值排行
	t = NewTable("服务接口平均响应时间排行", "排名", "响应时间", "服务名称", "接口", "访问量", "QPS")
	for i, p := range sm.GetAverageDurationPathList() {
		color := colorBlack
		if p.GetAverageDuration() > maxDuration {
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%v", p.GetAverageDuration()/1e6*1e6), //精度保留毫秒
			p.GetServiceName(),
			p.GetPath(),
			hrn(p.GetCount()),
			fmt.Sprintf("%.2f", p.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//响应时间最高的请求排行
	t = NewTable("响应时间最高的请求排行", "排名", "响应时间", "服务名称", "接口", "时间戳", "trace_id")
	for i, tp := range sm.GetMaxDurationTracePointList() {
		color := colorBlack
		if tp.Duration > maxDuration {
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%v", tp.Duration/1e6*1e6), //精度保留毫秒
			tp.ServiceName,
			tp.Path,
			tp.Timestamp.In(cstZone).Format(timeFormat),
			tp.TraceId,
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//常规cpu使用率排行
	t = NewTable("常规cpu使用率排行", "排名", "cpu使用率", "服务名称", "访问量", "平均响应时间", "QPS")
	for i, s := range sm.GetRemoveN100MaxCpuServiceList() {
		if s.GetRemoveN100MaxCpu() < cpuMin {
			break
		}
		color := colorBlack
		if s.GetRemoveN100MaxCpu() > 50 { //超过一半标红
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%d%%", s.GetRemoveN100MaxCpu()),
			s.Name(),
			hrn(s.GetCount()),
			fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
			fmt.Sprintf("%.2f", s.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//峰值cpu使用率排行
	t = NewTable("峰值cpu使用率排行", "排名", "cpu使用率", "服务名称", "访问量", "平均响应时间", "QPS")
	for i, s := range sm.GetMaxCpuServiceList() {
		if s.GetMaxCpu() < cpuMin {
			break
		}
		color := colorBlack
		if s.GetMaxCpu() > 80 { //超过80标红
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%d%%", s.GetMaxCpu()),
			s.Name(),
			hrn(s.GetCount()),
			fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
			fmt.Sprintf("%.2f", s.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//平均内存使用率排行
	t = NewTable("平均内存使用率排行", "排名", "内存使用率", "服务名称", "访问量", "平均响应时间", "QPS")
	for i, s := range sm.GetAverageMemServiceList() {
		if s.GetAverageMem() < memMin {
			break
		}
		color := colorBlack
		if s.GetAverageMem() > 50 { //超过50标红
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%d%%", s.GetAverageMem()),
			s.Name(),
			hrn(s.GetCount()),
			fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
			fmt.Sprintf("%.2f", s.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//峰值内存使用率排行
	t = NewTable("峰值内存使用率排行", "排名", "内存使用率", "服务名称", "访问量", "平均响应时间", "QPS")
	for i, s := range sm.GetMaxMemServiceList() {
		if s.GetMaxMem() < memMin {
			break
		}
		color := colorBlack
		if s.GetMaxMem() > 80 { //超过80标红
			color = colorRed
		}
		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%d%%", s.GetMaxMem()),
			s.Name(),
			hrn(s.GetCount()),
			fmt.Sprintf("%v", s.GetAverageDuration()/1e6*1e6),
			fmt.Sprintf("%.2f", s.GetQps()),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	//峰值磁盘使用率排行
	t = NewTable("峰值硬盘使用率排行", "排名", "硬盘使用率", "服务名称")
	for i, s := range sm.GetMaxDiskServiceList() {
		if s.GetMaxDisk() < diskMin {
			break
		}
		color := colorBlack
		if s.GetMaxDisk() > 80 {
			color = colorRed
		}

		_ = t.AddRecord(
			color,
			fmt.Sprintf("No.%d", i+1),
			fmt.Sprintf("%d%%", s.GetMaxDisk()),
			s.Name(),
		)
	}
	rtn.WriteString(t.ToHtml())
	rtn.WriteString("\n\n")

	rtn.WriteString("</pre></div>\n")

	return rtn.String()
}

type Line struct {
	color  string
	values []string
}

type Table struct {
	name       string
	keyList    []string
	recordList []Line
}

func NewTable(name string, keys ...string) *Table {
	return &Table{
		name:       name,
		keyList:    keys,
		recordList: make([]Line, 0),
	}
}

func (t *Table) AddRecord(color string, values ...string) error {
	if len(values) != len(t.keyList) {
		return errors.New("len(values) != len(keyList)")
	}

	t.recordList = append(t.recordList, Line{color: color, values: values})
	return nil
}

func (t *Table) ToHtml() string {
	rtn := new(strings.Builder)
	rtn.WriteString("<table>\n")
	rtn.WriteString("\t<style>table{border-spacing: 20px 0px}</style>\n")
	rtn.WriteString(fmt.Sprintf("\t<caption>%s</caption>\n", t.name))
	rtn.WriteString("\t<tr>")
	for _, k := range t.keyList {
		rtn.WriteString(fmt.Sprintf("<td>%s</td>", k))
	}
	rtn.WriteString("</tr>\n")
	for _, x := range t.recordList {
		rtn.WriteString("\t<tr>")
		for _, y := range x.values {
			rtn.WriteString(fmt.Sprintf("<td><font color=\"%s\">%s</font></td>", x.color, y))
		}
		rtn.WriteString("</tr>\n")
	}
	rtn.WriteString("</table>\n")
	return rtn.String()
}

/*
//添加记录
	t.recordList = append(t.recordList, valueList)
	return nil
}

//body := ServiceTableRun(service)
func SubTableRun(s *Service) string {
	rtn := new(strings.Builder)

	//标题
	rtn.WriteString("                服务健康状态表:" + s.Name() + "\n")

	//时间
	rtn.WriteString(fmt.Sprintf("时间：%s ～ %s\n",
		s.startTime.Format(time.RFC3339), s.endTime.Format(time.RFC3339)))

	//访问量
	rtn.WriteString(fmt.Sprintf("总访问量：%v  总响应时间：%v  平均响应时间：%v\n",
		s.GetCount(), s.GetSumDuration(), s.GetAverageDuration()))

	//基本信息
	rtn.WriteString(fmt.Sprintf("常规CPU使用率：%v  峰值CPU使用率：%v  峰值内存使用率：%v  峰值硬盘使用率：%v\n\n",
		s.GetRemoveN100MaxCpu(), s.GetMaxCpu(), s.GetMaxMem(), s.GetMaxDisk()))

	//接口中位响应时间排行
	medianDurationTable := NewTable("接口中位响应时间排行", "duration", "path", "traffic_volume")
	plist := s.GetMedianDurationPathList()
	for _, p := range plist {
		_ = medianDurationTable.AddRecord(p.GetMedianDuration(), p.GetPath(), p.GetCount())
	}
	rtn.WriteString(medianDurationTable.ToString())
	rtn.WriteString("\n")

	//接口平均响应时间排行
	averageDurationTable := NewTable("接口平均响应时间排行", "duration", "path", "traffic_volume")
	plist = s.GetAverageDurationPathList()
	for _, p := range plist {
		_ = averageDurationTable.AddRecord(p.GetAverageDuration(), p.GetPath(), p.GetCount())
	}
	rtn.WriteString(averageDurationTable.ToString())
	rtn.WriteString("\n")

	//接口峰值响应时间排行
	maxDurationTable := NewTable("接口峰值响应时间排行",
		"duration", "path", "timestamp", "trace_id")
	plist = s.GetAverageDurationPathList()
	for _, p := range plist {
		maxPoint := p.GetMaxDurationTracePoint()
		_ = maxDurationTable.AddRecord(maxPoint.Duration,
			maxPoint.Path, maxPoint.Timestamp, maxPoint.TraceId)
	}
	rtn.WriteString(maxDurationTable.ToString())
	rtn.WriteString("\n")

	return rtn.String()
}

func GeneralTableNewSM(n int) *ServiceMap {
	//求前n天的数据，按天取整
	now := time.Now()
	startTime := now.Add(-1 * time.Duration(now.Hour()) * time.Hour)
	startTime = startTime.Add(-1 * time.Duration(now.Minute()) * time.Minute)
	startTime = startTime.Add(-1 * time.Duration(now.Second()) * time.Second)
	startTime = startTime.Add(-1 * time.Duration(now.Nanosecond()))
	startTime = startTime.AddDate(0, 0, -n)
	endTime := startTime.AddDate(0, 0, n)

	//查询
	sm := NewServiceMap(startTime, endTime)

	return sm
}
*/
