package report_form

import (
	"encoding/json"
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"github.com/influxdata/influxdb/client/v2"
	"github.com/influxdata/influxdb/models"
	"strings"
	"time"
)

const (
	getServiceInfoSql = `SELECT * FROM %s WHERE sys_name = '%s' AND time >= %d AND time < %d;`
	tableTraceInfo    = "trace_info"
	minTrace          = 50 //小于50的调用不计入统计
	getServiceListSql = `show tag values from trace_info with key = sys_name`
	healthCheck       = `/tech/health/check`
)

//获取sql
func getSql(sysName string, table string, start time.Time, end time.Time) string {
	return fmt.Sprintf(getServiceInfoSql, table, sysName, start.UnixNano(), end.UnixNano())
}

//获取row的index
func getModRowKeyIdx(key string, row models.Row) (int, bool) {
	for i, v := range row.Columns {
		if v == key {
			return i, true
		}
	}

	return -1, false
}

//查询
func Query(sql string) (*client.Response, error) {
	config := client.HTTPConfig{
		Addr:    global.InfluxDbAddress,
		Timeout: time.Second * 30,
	}

	connect, err := client.NewHTTPClient(config)
	defer func() { _ = connect.Close() }()
	if err != nil {
		return nil, err
	}

	rtn, err := connect.Query(client.NewQuery(sql, global.InfluxDbName, ""))
	if err != nil {
		return nil, err
	}

	return rtn, nil
}

//从db获取服务列表
func GetServiceNameListFromDb() []string {
	resp, err := Query(getServiceListSql)
	if err != nil {
		return []string{}
	}

	rtn := make([]string, 0)
	for _, v := range resp.Results {
		for _, row := range v.Series {
			index, ok := getModRowKeyIdx("value", row)
			if !ok {
				continue
			}
			for _, value := range row.Values {
				serviceName, ok := value[index].(string)
				if !ok {
					continue
				}
				rtn = append(rtn, serviceName)
			}
		}
	}
	return rtn
}

//从db拉取数据
func GetDataFromDb(startTime time.Time, endTime time.Time) ServiceMap {
	serviceNameList := GetServiceNameListFromDb()

	serviceMap := NewServiceMap()
	for _, serviceName := range serviceNameList {
		sql := getSql(serviceName, tableTraceInfo, startTime, endTime)
		resp, err := Query(sql)
		if err != nil {
			continue
		}

		for _, v := range resp.Results {
			for _, row := range v.Series {
				traceList := rowToTraceList(serviceName, row)
				for _, trace := range traceList {
					serviceMap.AddTrace(serviceName, trace.Path, trace)
				}
			}
		}
	}

	return serviceMap
}

func rowToTraceList(serviceName string, row models.Row) []*Trace {
	//[time duration host msg path sys_name traceId]
	rtn := make([]*Trace, 0)
	durationIndex, ok := getModRowKeyIdx("duration", row)
	if !ok {
		return rtn
	}
	pathIndex, ok := getModRowKeyIdx("path", row)
	if !ok {
		return rtn
	}
	timeIndex, ok := getModRowKeyIdx("time", row)
	if !ok {
		return rtn
	}
	idIndex, ok := getModRowKeyIdx("traceId", row)
	if !ok {
		return rtn
	}
	hostIndex, ok := getModRowKeyIdx("host", row)
	if !ok {
		return rtn
	}

	for _, value := range row.Values {
		durationRaw, ok := value[durationIndex].(json.Number)
		if !ok {
			continue
		}
		durationInt64, err := durationRaw.Int64()
		if err != nil {
			continue
		}
		duration := time.Duration(durationInt64) * 1000000 //精度为毫秒，*1000后为纳秒
		path, ok := value[pathIndex].(string)
		if !ok {
			continue
		}
		pathSplit := strings.Split(path, " ")
		if len(pathSplit) != 2 || pathSplit[1] == healthCheck {
			continue
		}
		timeRaw, ok := value[timeIndex].(string)
		if !ok {
			continue
		}
		timePoint, err := time.Parse(time.RFC3339, timeRaw)
		if err != nil {
			continue
		}
		id, ok := value[idIndex].(string)
		if !ok {
			continue
		}
		host := value[hostIndex].(string)
		if !ok {
			continue
		}

		rtn = append(rtn, NewTrace(id, serviceName, host, path, duration, timePoint))
	}

	return rtn
}
