package points

import (
	"fmt"
	"strings"
	"sync"
)

//服务A 的 X 接口 被 那些服务访问？ 那些节点访问
type PathNode struct {
	ServiceName      string
	Path             string
	RemoteServiceMap map[string]struct{}
}

func NewPathNode(path string, serviceName string) PathNode {
	return PathNode{
		Path:             path,
		ServiceName:      serviceName,
		RemoteServiceMap: make(map[string]struct{}),
	}
}

type ServiceNode struct {
	ServiceName string
	PathMap     map[string]PathNode
	IpMap       map[string]struct{}
}

func NewServiceNode(serviceName string) ServiceNode {
	return ServiceNode{
		ServiceName: serviceName,
		PathMap:     make(map[string]PathNode),
		IpMap:       make(map[string]struct{}),
	}
}

type Statistics struct {
	ServiceMap map[string]ServiceNode
	lock       *sync.Mutex
}

func (s *Statistics) String() string {
	s.lock.Lock()
	defer s.lock.Unlock()
	rtn := ""

	for _, service := range s.ServiceMap {
		rtn += fmt.Sprintf("%s: ", service.ServiceName)
		for ip := range service.IpMap {
			rtn += ip + "  "
		}
		rtn += "\n"

		for _, path := range service.PathMap {
			rtn += path.Path + "  "
			for remoteService := range path.RemoteServiceMap {
				rtn += remoteService + "  "
			}
			rtn += "\n"
		}

		rtn += "\n"
	}

	return rtn
}

func NewStatistics() *Statistics {
	return &Statistics{
		ServiceMap: make(map[string]ServiceNode),
		lock:       new(sync.Mutex),
	}
}

func (s *Statistics) addServer(point TraceMsg) {
	local := point.LocalEndpoint
	//remote := point.RemoteEndpoint
	tag := point.Tags

	if _, ok := s.ServiceMap[local.ServiceName]; !ok {
		s.ServiceMap[local.ServiceName] = NewServiceNode(local.ServiceName)
	}

	if _, ok := s.ServiceMap[local.ServiceName].IpMap[local.Ipv4]; !ok {
		s.ServiceMap[local.ServiceName].IpMap[local.Ipv4] = struct{}{}
	}

	path := strings.ToLower(point.Name)
	if _, ok := methodMap[strings.ToLower(point.Name)]; ok {
		path = strings.ToLower(tag.HttpMethod + "-" + tag.HttpPath)
	}
	_, ok := s.ServiceMap[local.ServiceName].PathMap[path]
	if !ok {
		s.ServiceMap[local.ServiceName].PathMap[path] = NewPathNode(path, local.ServiceName)
	}
}

func (s *Statistics) addClient(point TraceMsg) {
	local := point.LocalEndpoint
	remote := point.RemoteEndpoint
	tag := point.Tags
	for sName := range s.ServiceMap {
		for ip := range s.ServiceMap[sName].IpMap {
			if ip == remote.Ipv4 {
				//添加服务
				path := strings.ToLower(point.Name)
				if _, ok := methodMap[strings.ToLower(point.Name)]; ok {
					path = strings.ToLower(tag.HttpMethod + "-" + tag.HttpPath)
				}
				if _, ok := s.ServiceMap[sName].PathMap[path]; !ok {
					s.ServiceMap[sName].PathMap[path] = NewPathNode(path, s.ServiceMap[sName].ServiceName)
				}

				s.ServiceMap[sName].PathMap[path].RemoteServiceMap[local.ServiceName] = struct{}{}
			}
		}
	}
}

func (s *Statistics) AddNode(point TraceMsg) {
	s.lock.Lock()
	defer s.lock.Unlock()

	switch point.Kind {
	case "SERVER":
		s.addServer(point)
	case "CLIENT":
		s.addClient(point)
	default:
		//位置类型，不处理
	}
}

/*
type PathNode struct {
	ServiceName    string
	Path           string
	SrcServiceList map[string]struct{}
	SrcNodeIp      map[string]struct{}
}

type Statistics struct {
	PathMap map[string]PathNode
	lock    *sync.Mutex
}

func NewStatistics() *Statistics {
	return &Statistics{
		PathMap: make(map[string]PathNode),
		lock:    new(sync.Mutex),
	}
}

*/
