package main

import (
	"fmt"
	"git.quantgroup.cn/DevOps/enoch/pkg/api-server"
	"git.quantgroup.cn/DevOps/enoch/pkg/dao"
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"git.quantgroup.cn/DevOps/enoch/pkg/glog"
	node_check "git.quantgroup.cn/DevOps/enoch/pkg/node-check"
	_ "git.quantgroup.cn/DevOps/enoch/pkg/node-up-down"
	"git.quantgroup.cn/DevOps/enoch/pkg/points"
	"git.quantgroup.cn/DevOps/enoch/pkg/report-form"
	"github.com/Shopify/sarama"
	_ "github.com/mkevac/debugcharts"
	"github.com/valyala/fasthttp"
	"net/http"
	_ "net/http/pprof"
	"os"
	"os/signal"
	"strings"
	"syscall"
	"time"
)

func handlerKafkaMsg() {
	fmt.Println(global.InfluxDbAddress)
	db := dao.New(global.BatchSize, time.Second*60, global.InfluxDbAddress, global.DaoFileCacheDir)

	//处理调用链条信息
	braveTopic := global.Config.GetOrDefault(global.NamespaceTechSleuth, "tech.brave.kafkaTopic", "")

	braveConsumer, err := global.KafkaRecver.NewConsumer("enoch-group", []string{braveTopic}, func(msg *sarama.ConsumerMessage) {
		pointList, err := points.TraceBaseInfoToPoint(msg.Value)
		if err != nil {
			glog.Errorf("can not make trace point:", err)
			return
		}
		for _, point := range pointList {
			db.MsgProcess(point)
		}
	})
	if err != nil {
		glog.Fatal("监听kafka recver失败!")
	}

	//处理基本信息
	healthTopic := global.Config.GetOrDefault(global.NamespaceTechSleuth, "tech.enoch.kafkaTopic", "")
	healthConsumer, err := global.KafkaRecver.NewConsumer("quantGroup-enoch-agent", []string{healthTopic}, func(msg *sarama.ConsumerMessage) {
		pointList, err := points.HostBaseInfoToPoint(msg.Value)
		if err != nil {
			glog.Errorf("can not make health point:", err)
			return
		}
		for _, point := range pointList {
			db.MsgProcess(point)
		}
	})
	if err != nil {
		glog.Fatal("监听kafka recver失败!")
	}

	glog.Info(global.AppName + "启动")
	//平滑退出
	sigterm := make(chan os.Signal, 1)
	signal.Notify(sigterm, syscall.SIGINT, syscall.SIGTERM)
	select {
	case <-sigterm:
		if braveConsumer != nil {
			braveConsumer.Close()
			glog.Info("braceConsumer平滑退出")
		}
		if healthConsumer != nil {
			healthConsumer.Close()
			glog.Info("healthConsumer平滑退出")
		}
		if db != nil {
			db.Close()
			glog.Info("dao平滑退出")
		}
		glog.Info(global.AppName + "平滑退出")
	}
}

//TODO 可优化成Raft算法，目前是固定节点的
func isMaster() bool {
	//开发环境，但InfluxDb的Ip是生产环境ip的时候，不执行初始化操作
	if global.IsDev() && strings.Contains(global.InfluxDbAddress, "172.20") {
		return false
	}

	//生产环境，但LocalIp不是"172.30.12.22"，不执行初始化操作
	if !global.IsDev() && global.LocalIp != global.MasterAddress {
		return false
	}

	return true
}

func main() {
	//性能监控
	go func() {
		defer func() {
			if err := recover(); err != nil {
				glog.Error(err)
			}
		}()
		if err := http.ListenAndServe(":9999", nil); err != nil {
			glog.Error(err)
		}
	}()

	//主节点：刷新连续查询、健康状态报表、告警策略
	if isMaster() {
		//初始化数据库（创建DB，创建连续查询）
		dao.DbInit()

		//健康状态报表
		//每天0点生成报表
		report_form.RegularReport(global.ReportFormDir)
		//每周1早10点发送邮件
		report_form.RegularMail(global.ReportFormDir)
		//节点健康状态检查
		node_check.NodeHealthCheckAndNotify()
		//TODO 告警策略

	}

	//对外api
	api_server.ListenAndServe()

	//time.Sleep(time.Second * 100000)
	//创建server

	//处理消息（阻塞）
	handlerKafkaMsg()

	//阻塞main
	//select {}
}

func init() {
	api_server.HandlerFunc(api_server.GET, "/tech/health/check", healthCheck)
	api_server.HandlerFunc(api_server.HEAD, "/tech/health/check", healthCheck)

}

func healthCheck(ctx *fasthttp.RequestCtx) {
	ctx.Response.SetStatusCode(fasthttp.StatusOK)
	ctx.Response.SetBodyString("ok")
}
