package report_form

import (
	"errors"
	"fmt"
	"strings"
	"time"
	"unicode/utf8"
)

type Table struct {
	name            string
	keyList         []string
	maxLenValueList []int
	recordList      [][]string
}

func NewTable(name string, keys ...string) *Table {
	maxLenValueList := make([]int, len(keys))
	for i, key := range keys {
		maxLenValueList[i] = strFormatLen(key)
	}

	return &Table{
		name:            name,
		keyList:         keys,
		recordList:      make([][]string, 0),
		maxLenValueList: maxLenValueList,
	}
}

func strFormatLen(s string) int {
	l := 0
	for i := 0; i < len(s); {
		_, size := utf8.DecodeRuneInString(s[i:])
		if size != 1 {
			l += 2
		} else {
			l++
		}
		i += size
	}
	return l
}

func (t *Table) ToString() string {
	const spacing = 4
	rtn := new(strings.Builder)

	//标题
	totalLength := spacing * (len(t.maxLenValueList) - 1)
	for _, valueLen := range t.maxLenValueList {
		totalLength += valueLen
	}
	preBlankLen := (totalLength - strFormatLen(t.name)) / 2
	postBlankLen := totalLength - preBlankLen - strFormatLen(t.name)

	for i := 0; i < preBlankLen; i++ {
		rtn.WriteString("-")
	}
	rtn.WriteString(t.name)
	for i := 0; i < postBlankLen; i++ {
		rtn.WriteString("-")
	}
	rtn.WriteString("\n")

	//key
	for i, key := range t.keyList {
		rtn.WriteString(key)

		//补全空格，最后一行结尾不打印分隔空白
		blankLen := t.maxLenValueList[i] - strFormatLen(key) + spacing
		if i == len(t.keyList)-1 {
			blankLen -= spacing
		}
		for i := 0; i < blankLen; i++ {
			rtn.WriteString(" ")
		}
	}
	rtn.WriteString("\n")

	//value
	for _, valueList := range t.recordList {
		for i, value := range valueList {
			rtn.WriteString(value)

			blankLen := t.maxLenValueList[i] - strFormatLen(value) + spacing
			if i == len(t.keyList)-1 {
				blankLen -= spacing
			}
			for i := 0; i < blankLen; i++ {
				rtn.WriteString(" ")
			}
		}
		rtn.WriteString("\n")
	}

	return rtn.String()
}

//添加记录
func (t *Table) AddRecord(values ...interface{}) error {
	if len(values) != len(t.keyList) {
		return errors.New("len(values) != len(keyList)")
	}

	//interface转string，并且记录最大长度
	valueList := make([]string, 0)
	for i, v := range values {
		value := fmt.Sprint(v)
		valueList = append(valueList, value)
		if strFormatLen(value) > t.maxLenValueList[i] {
			t.maxLenValueList[i] = strFormatLen(value)
		}
	}

	t.recordList = append(t.recordList, valueList)
	return nil
}

func Run(n int) string {
	const (
		cpuMin  = 1
		memMin  = 1
		diskMin = 1
	)

	//求前n天的数据，按天取整
	now := time.Now()
	startTime := now.Add(-1 * time.Duration(now.Hour()) * time.Hour)
	startTime = startTime.Add(-1 * time.Duration(now.Minute()) * time.Minute)
	startTime = startTime.Add(-1 * time.Duration(now.Second()) * time.Second)
	startTime = startTime.Add(-1 * time.Duration(now.Nanosecond()))
	startTime = startTime.AddDate(0, 0, -n)
	endTime := startTime.AddDate(0, 0, n)

	//查询
	sm := NewServiceMap(startTime, endTime)

	//生成报表
	rtn := new(strings.Builder)

	//表头
	rtn.WriteString("                        服务健康状态总表\n")

	//时间
	rtn.WriteString(fmt.Sprintf("时间：%s ～ %s\n", startTime.Format(time.RFC3339), endTime.Format(time.RFC3339)))

	//访问量
	rtn.WriteString(fmt.Sprintf("总访问量：%v  总响应时间：%v  平均响应时间：%v\n\n",
		sm.GetCount(), sm.GetSumDuration(), sm.GetAverageDuration()))

	//服务响应时间中位值排行
	medianTable := NewTable("服务响应时间中位值排行", "duration", "sys_name", "path", "count")
	medianList := sm.GetMedianDurationPathList()
	for _, m := range medianList {
		_ = medianTable.AddRecord(m.GetMedianDuration(), m.GetServiceName(), m.GetPath(), m.GetCount())
	}
	rtn.WriteString(medianTable.ToString())
	rtn.WriteString("\n")

	//服务响应时间平均值排行
	averageTable := NewTable("服务响应时间平均值排行", "duration", "sys_name", "path", "count")
	averageList := sm.GetAverageDurationPathList()
	for _, m := range averageList {
		_ = averageTable.AddRecord(m.GetAverageDuration(), m.GetServiceName(), m.GetPath(), m.GetCount())
	}
	rtn.WriteString(averageTable.ToString())
	rtn.WriteString("\n")

	//响应时间最长的请求排行
	maxTable := NewTable("响应时间最长的请求排行", "duration", "sys_name", "path", "timestamp", "trace_id")
	maxList := sm.GetMaxDurationTracePointList()
	for _, m := range maxList {
		_ = maxTable.AddRecord(m.Duration, m.ServiceName, m.Path, m.Timestamp, m.TraceId)
	}
	rtn.WriteString(maxTable.ToString())
	rtn.WriteString("\n")

	//平均cpu使用率排行
	cpuTable := NewTable("平均cpu使用率排行", "cpu", "sys_name", "count", "average_duration")
	cpuList := sm.GetAverageCpuServiceList()
	for _, s := range cpuList {
		if s.GetAverageCpu() < cpuMin {
			break
		}
		_ = cpuTable.AddRecord(s.GetAverageCpu(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(cpuTable.ToString())
	rtn.WriteString("\n")

	//平均内存使用率排行
	memTable := NewTable("平均内存用率排行", "mem", "sys_name", "count", "average_duration")
	memList := sm.GetAverageMemServiceList()
	for _, s := range memList {
		if s.GetAverageMem() < memMin {
			break
		}
		_ = memTable.AddRecord(s.GetAverageMem(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(memTable.ToString())
	rtn.WriteString("\n")

	//平均硬盘使用率排行
	diskTable := NewTable("平均硬盘使用率排行", "disk", "sys_name", "count", "average_duration")
	diskList := sm.GetAverageDiskServiceList()
	for _, s := range diskList {
		if s.GetAverageDisk() < diskMin {
			break
		}
		_ = diskTable.AddRecord(s.GetAverageDisk(), s.Name(), s.GetCount(), s.GetAverageDuration())
	}
	rtn.WriteString(diskTable.ToString())
	rtn.WriteString("\n")

	return rtn.String()
}
