package points

import (
	"git.quantgroup.cn/DevOps/enoch/pkg/global"
	"github.com/vrg0/go-common/logger"
	"go.uber.org/zap/zapcore"
	"sync"
	"time"
)

type Node struct {
	CreateTime time.Time
	Points     TraceMsg
}

type Delayed5s struct {
	Server map[string]Node
	Client map[string]Node
	lock   *sync.Mutex
	save   *logger.Logger
}

func NewDelayed5s() *Delayed5s {
	savePath := global.LogPath
	if savePath != "/dev/stdout" {
		savePath += "_delayed5s"
	}

	return &Delayed5s{
		Server: make(map[string]Node),
		Client: make(map[string]Node),
		lock:   new(sync.Mutex),
		save:   logger.New(savePath, zapcore.DebugLevel),
	}
}

func (d *Delayed5s) Run() {
	d.save.Info("5s延时监控程序启动")
	go func() {
		for {
			time.Sleep(time.Second * 30)

			now := time.Now()
			d.lock.Lock()

			//1、删除过期的client
			for id, node := range d.Client {
				if now.Unix() > node.CreateTime.Add(time.Minute * 10).Unix() {
					delete(d.Client, id)
				}
			}

			//2、删除过期的server
			for id, node := range d.Server {
				if now.Unix() > node.CreateTime.Add(time.Minute * 10).Unix() {
					delete(d.Server, id)
				}
			}

			//3、记录&删除延时请求
			for id, clientNode := range d.Client {
				serverNode, ok := d.Server[id]
				if !ok {
					continue
				}

				if clientNode.Points.Duration-serverNode.Points.Duration > int(1*time.Second) {
					d.save.Info(clientNode.Points, serverNode.Points)
					delete(d.Client, id)
					delete(d.Server, id)
				}
			}

			d.lock.Unlock()
		}
	}()
}

func (d *Delayed5s) AddNode(point TraceMsg) {
	d.lock.Lock()
	defer d.lock.Unlock()

	switch point.Kind {
	case "SERVER":
		d.Server[point.Id] = Node{time.Now(), point}
	case "CLIENT":
		d.Client[point.Id] = Node{time.Now(), point}
	default:
		//位置类型，不处理
	}
}
