package cn.quantgroup.boot.tech.brave.filter;

import brave.servlet.TracingFilter;
import brave.spring.webmvc.DelegatingTracingFilter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpMethod;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;

/**
 * Similar to {@link TracingFilter}, except that it initializes from Spring.
 *
 * <p> {@link org.springframework.web.filter.DelegatingFilterProxy} is similar, but it uses
 * volatile references as it allows lazy initialization from doGet. This filter cannot do that
 * anyway because {@code ServletRequest.getServletContext()} was added after servlet 2.5!
 */
@Slf4j
public final class CustomDelegatingTracingFilter implements Filter {

    // servlet ensures create is directly followed by init, so no need for volatile
    DelegatingTracingFilter delegatingTracingFilter;

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        HttpServletRequest httpServletRequest = (HttpServletRequest) request;

        // don't break on initialization error.
        if (HttpMethod.OPTIONS.name().equals(httpServletRequest.getMethod()) || delegatingTracingFilter == null) {
            chain.doFilter(request, response);
        } else {
            delegatingTracingFilter.doFilter(request, response, chain);
        }
    }

    @Override
    public void init(FilterConfig filterConfig) {
        DelegatingTracingFilter delegatingTracingFilter = new DelegatingTracingFilter();
        delegatingTracingFilter.init(filterConfig);
        this.delegatingTracingFilter = delegatingTracingFilter;
    }

    @Override
    public void destroy() {
        // TracingFilter is stateless, so nothing to destroy
    }
}
