package cn.quantgroup.boot;

import com.alibaba.cloud.commons.lang.StringUtils;
import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.balancer.NacosBalancer;
import com.alibaba.cloud.nacos.loadbalancer.NacosLoadBalancer;
import com.alibaba.nacos.common.utils.CollectionUtils;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.loadbalancer.DefaultResponse;
import org.springframework.cloud.client.loadbalancer.EmptyResponse;
import org.springframework.cloud.client.loadbalancer.Request;
import org.springframework.cloud.client.loadbalancer.Response;
import org.springframework.cloud.loadbalancer.core.NoopServiceInstanceListSupplier;
import org.springframework.cloud.loadbalancer.core.ReactorServiceInstanceLoadBalancer;
import org.springframework.cloud.loadbalancer.core.ServiceInstanceListSupplier;
import reactor.core.publisher.Mono;

public class VersionGrayLoadBalancer implements ReactorServiceInstanceLoadBalancer {

  private static final Logger log = LoggerFactory.getLogger(NacosLoadBalancer.class);

  private final String serviceId;

  ObjectProvider<ServiceInstanceListSupplier> serviceInstanceListSupplierProvider;
  private final NacosDiscoveryProperties nacosDiscoveryProperties;

  public VersionGrayLoadBalancer(
      ObjectProvider<ServiceInstanceListSupplier> serviceInstanceListSupplierProvider,
      String serviceId, NacosDiscoveryProperties nacosDiscoveryProperties) {
    this.serviceId = serviceId;
    this.serviceInstanceListSupplierProvider = serviceInstanceListSupplierProvider;
    this.nacosDiscoveryProperties = nacosDiscoveryProperties;
  }


  @Override
  public Mono<Response<ServiceInstance>> choose(Request request) {
    ServiceInstanceListSupplier supplier = serviceInstanceListSupplierProvider
        .getIfAvailable(NoopServiceInstanceListSupplier::new);
    return supplier.get().next()
        .mapNotNull(this::getInstanceResponse);
  }

  private Response<ServiceInstance> getInstanceResponse(
      List<ServiceInstance> serviceInstances) {
    if (serviceInstances.isEmpty()) {
      log.warn("No servers available for service: " + this.serviceId);
      return new EmptyResponse();
    }

    try {
      String clusterName = this.nacosDiscoveryProperties.getClusterName();
      List<ServiceInstance> instancesToChoose = serviceInstances;
      if (StringUtils.isNotBlank(clusterName)) {

        List<ServiceInstance> sameClusterInstances = serviceInstances.stream()
            .filter(serviceInstance -> {
              String cluster = serviceInstance.getMetadata()
                  .get("nacos.cluster");
              return StringUtils.equals(cluster, clusterName);
            }).collect(Collectors.toList());

        String version = MDC.get("x-b3-version");

        if (StringUtils.isEmpty(version)) {
          sameClusterInstances = sameClusterInstances.stream()
              .filter(i -> StringUtils.isEmpty(i.getMetadata().get("version"))).collect(
                  Collectors.toList());
        } else {
          List<ServiceInstance> grayInstances = sameClusterInstances.stream()
              .filter(i -> version.equals(i.getMetadata().get("version"))).collect(
                  Collectors.toList());
          if (CollectionUtils.isNotEmpty(grayInstances)) {
            sameClusterInstances = grayInstances;
          }

        }

        instancesToChoose = sameClusterInstances;
      } else {
        log.warn(
            "A cross-cluster call occurs，name = {}, clusterName = {}, instance = {}",
            serviceId, clusterName, serviceInstances);
      }

      ServiceInstance instance = NacosBalancer
          .getHostByRandomWeight3(instancesToChoose);

      return new DefaultResponse(instance);
    } catch (Exception e) {
      log.warn("NacosLoadBalancer error", e);
      return null;
    }
  }

}
