package cn.quantgroup.tech.brave.configuration;

import brave.Tracing;
import brave.context.log4j2.ThreadContextCurrentTraceContext;
import brave.http.HttpTracing;
import brave.httpclient.TracingHttpClientBuilder;
import brave.okhttp3.TracingInterceptor;
import brave.propagation.B3Propagation;
import brave.propagation.ExtraFieldPropagation;
import brave.spring.rabbit.SpringRabbitTracing;
import brave.spring.web.TracingClientHttpRequestInterceptor;
import brave.spring.webmvc.DelegatingTracingFilter;
import brave.spring.webmvc.SpanCustomizingAsyncHandlerInterceptor;
import cn.quantgroup.tech.brave.properties.BraveProperties;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;
import org.apache.http.impl.client.CloseableHttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.EnableWebMvc;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import zipkin2.Span;
import zipkin2.codec.Encoding;
import zipkin2.reporter.AsyncReporter;
import zipkin2.reporter.Sender;
import zipkin2.reporter.kafka11.KafkaSender;

import javax.annotation.PostConstruct;
import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.List;

/**
 * 注册brave
 *
 * @author zhangheng
 * create on 2018.04.25
 */
@Slf4j
@Configuration
@AutoConfigureOrder
@EnableWebMvc
@EnableConfigurationProperties( BraveProperties.class )
@Import( {
        TracingClientHttpRequestInterceptor.class,
        SpanCustomizingAsyncHandlerInterceptor.class} )
public class BraveAutoConfiguration extends WebMvcConfigurerAdapter {
    @Autowired
    private BraveProperties braveProperties;

    @Autowired
    SpanCustomizingAsyncHandlerInterceptor spanCustomizingAsyncHandlerInterceptor;

    @Autowired
    TracingClientHttpRequestInterceptor clientInterceptor;

    @Bean
    Sender sender() {
        return KafkaSender.newBuilder().bootstrapServers(braveProperties.getKafkaHost()).topic(braveProperties.getKafkaTopic()).encoding(Encoding.JSON).build();
    }

    @Bean
    AsyncReporter<Span> spanReporter() {
        return AsyncReporter.create(sender());
    }

    @Bean
    Tracing tracing() {
        return Tracing.newBuilder()
                .sampler(brave.sampler.Sampler.create(braveProperties.getSample()))
                .localServiceName(braveProperties.getServiceName())
                .propagationFactory(ExtraFieldPropagation.newFactory(B3Propagation.FACTORY, "user-name"))
                .currentTraceContext(ThreadContextCurrentTraceContext.create()) // puts trace IDs into logs
                .spanReporter(spanReporter()).build();
    }

    @Bean
    public HttpTracing httpTracing(Tracing tracing) {
        return HttpTracing.create(tracing);
    }

    @Bean
    public SpringRabbitTracing springRabbitTracing(Tracing tracing) {
        return SpringRabbitTracing.newBuilder(tracing)
                .remoteServiceName("my-mq-service").build();
    }

    @Bean( name = "techHttpClient" )
    public CloseableHttpClient httpClient(Tracing tracing) {
        log.info("注册HttpClient");
        return TracingHttpClientBuilder.create(tracing).build();
    }

    @Bean( name = "techOkHttpClient" )
    public OkHttpClient httpClient(HttpTracing httpTracing) {
        log.info("注册OkHttpClient");
        return new OkHttpClient.Builder()
                .dispatcher(new Dispatcher(httpTracing.tracing().currentTraceContext().executorService(new Dispatcher().executorService())))
                .addNetworkInterceptor(TracingInterceptor.create(httpTracing)).build();
    }

    /**
     * 创建一个bean
     *
     * @return
     */
    @Bean
    public Filter delegatingTracingFilter() {
        return new DelegatingTracingFilter();
    }

    @Bean
    public FilterRegistrationBean someFilterRegistration(Filter delegatingTracingFilter) {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(delegatingTracingFilter);
        registration.addUrlPatterns("/*");
        registration.setName("DelegatingTracingFilter");
        return registration;
    }

    /**
     * adds tracing to the application-defined rest template
     */
    @PostConstruct
    public void init() {
        log.info("添加restTemplate拦截器");
        RestTemplate restTemplate = new RestTemplate();
        List<ClientHttpRequestInterceptor> interceptors =
                new ArrayList<>(restTemplate.getInterceptors());
        interceptors.add(clientInterceptor);
        restTemplate.setInterceptors(interceptors);
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        log.info("添加webmvc拦截器");
        registry.addInterceptor(spanCustomizingAsyncHandlerInterceptor);
    }
}


