package cn.quantgroup.boot;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.registry.NacosAutoServiceRegistration;
import feign.Feign;
import feign.RequestInterceptor;
import org.slf4j.MDC;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.boot.actuate.endpoint.web.annotation.WebEndpoint;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.EnableDiscoveryClient;
import org.springframework.cloud.client.loadbalancer.LoadBalanced;
import org.springframework.cloud.loadbalancer.annotation.LoadBalancerClient;
import org.springframework.cloud.loadbalancer.annotation.LoadBalancerClients;
import org.springframework.cloud.loadbalancer.core.ReactorLoadBalancer;
import org.springframework.cloud.loadbalancer.core.ServiceInstanceListSupplier;
import org.springframework.cloud.loadbalancer.support.LoadBalancerClientFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.core.env.Environment;
import org.springframework.web.filter.CommonsRequestLoggingFilter;

@Configuration
@DependsOn({"nacosServiceDiscovery"})
@EnableDiscoveryClient
public class DisCoveryAutoConfiguration {

  final NacosAutoServiceRegistration nacosAutoServiceRegistration;

  public DisCoveryAutoConfiguration(NacosAutoServiceRegistration nacosAutoServiceRegistration) {
    this.nacosAutoServiceRegistration = nacosAutoServiceRegistration;
  }

  @LoadBalancerClients({
      @LoadBalancerClient(value = "price-tracker", configuration = CustomLoadBalancerConfiguration.class),
      @LoadBalancerClient(value = "sc-task", configuration = CustomLoadBalancerConfiguration.class),
      @LoadBalancerClient(value = "sc-auth", configuration = CustomLoadBalancerConfiguration.class)}
  )
  public class FeignConfig {

    @LoadBalanced
    @Bean
    public Feign.Builder feignBuilder() {
      return Feign.builder();
    }

    @Bean
    public RequestInterceptor requestInterceptor() {
      return requestTemplate -> requestTemplate.header("x-b3-version", MDC.get("x-b3-version"));
    }
  }

  @Bean
  public RequestLoggingFilter requestLoggingFilter() {
    return new RequestLoggingFilter();
  }
  @Bean
  public CommonsRequestLoggingFilter logFilter(RequestLoggingFilter requestLoggingFilter) {
    requestLoggingFilter.setIncludeQueryString(true);
    requestLoggingFilter.setIncludePayload(true);
    requestLoggingFilter.setMaxPayloadLength(10000);
    requestLoggingFilter.setIncludeHeaders(false);
    requestLoggingFilter.setAfterMessagePrefix("REQUEST DATA : ");
    return requestLoggingFilter;
  }


  public static class CustomLoadBalancerConfiguration {

    @Bean
    ReactorLoadBalancer<ServiceInstance> versionGrayLoadBalancer(Environment environment,
        LoadBalancerClientFactory loadBalancerClientFactory,
        NacosDiscoveryProperties nacosDiscoveryProperties) {
      String name = environment.getProperty(LoadBalancerClientFactory.PROPERTY_NAME);
      return new VersionGrayLoadBalancer(
          loadBalancerClientFactory.getLazyProvider(name, ServiceInstanceListSupplier.class), name,
          nacosDiscoveryProperties);
    }
  }
}
