package cn.quantgroup.tech.enoch.agent.report.impl;

import cn.quantgroup.tech.enoch.agent.configuration.EnochAgentProperties;
import cn.quantgroup.tech.enoch.agent.report.IAgentReportService;
import com.google.gson.Gson;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.env.EnvironmentEndpoint;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthComponent;
import org.springframework.boot.actuate.health.HealthEndpoint;
import org.springframework.boot.actuate.metrics.MetricsEndpoint;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.stream.Collectors;

/**
 * @author: amen
 * @date: 2019-02-26
 */
@Slf4j
@Service
public class AgentReportServiceImpl implements IAgentReportService {


    @Autowired
    private HealthEndpoint healthMvcEndpoint;

    @Autowired
    private MetricsEndpoint metricsEndpoint;

    @Autowired
    private KafkaTemplate<String, String> kafkaTemplate;

    @Autowired
    private EnochAgentProperties enochAgentProperties;

    @Autowired
    private EnvironmentEndpoint environmentEndpoint;

    @Autowired
    private Gson gson;


    private final ArrayBlockingQueue<Map<String, Object>> arrayBlockingQueue = new ArrayBlockingQueue<>(600);

    private final ApplicationInfo applicationInfo = new ApplicationInfo();

    @Override
    public void report() {
        int size = arrayBlockingQueue.size();
        if (size <= 0) {
            return;
        }
        List<Map<String, Object>> slice = new ArrayList<>(size);
        arrayBlockingQueue.drainTo(slice, size);
        Map<String, Object> pushPackage = new HashMap<>(6);
        if (StringUtils.isBlank(applicationInfo.getAppName())) {
            initApplicationInfo();
        }
        pushPackage.put("ip", applicationInfo.getIp());
        pushPackage.put("appName", applicationInfo.getAppName());
        pushPackage.put("endPoints", slice);
        //todo 若发kafka失败，添加本地持久化操作，待下次提交时，一同push

        kafkaTemplate.send(enochAgentProperties.getKafkaTopic(), gson.toJson(pushPackage));
    }

    @Override
    public void collect() {
        HashMap<String, Object> endPoints = new HashMap<>(16);

        HealthComponent health =healthMvcEndpoint.health();
        Map<String, List<MetricsEndpoint.Sample>> metricCollect = metricsEndpoint.listNames().getNames().stream().collect(Collectors.toMap(i -> i, j -> metricsEndpoint.metric(j, null).getMeasurements()));

        Health.status(health.getStatus()).build();
        endPoints.put("health", health);
        endPoints.put("metrics", metricCollect);
        endPoints.put("timestamp", System.currentTimeMillis());
        try {
            arrayBlockingQueue.put(endPoints);
        } catch (InterruptedException e) {
            log.error("enoch.agent collect data error: {}", e);
        }
    }

    private ApplicationInfo initApplicationInfo() {
        String appName = (String) environmentEndpoint.environmentEntry("spring.application.name").getProperty().getValue();
        String ip =(String) environmentEndpoint.environmentEntry("spring.cloud.client.ipAddress").getProperty().getValue();
        applicationInfo.setAppName(appName);
        applicationInfo.setIp(ip);
        return applicationInfo;
    }

    @Data
    public static class ApplicationInfo {
        private String ip;
        private String appName;
    }
}
