package cn.quantgroup.tech.mq.config;

import cn.quantgroup.tech.mq.common.Constants;
import cn.quantgroup.tech.mq.entity.KoalaRabbitmqMessage;
import cn.quantgroup.tech.mq.mapper.KoalaRabbitmqMessageMapperV1;
import cn.quantgroup.tech.mq.service.MONotify;
import org.springframework.amqp.AmqpException;
import org.springframework.amqp.core.AcknowledgeMode;
import org.springframework.amqp.core.CorrelationAwareMessagePostProcessor;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.rabbit.support.CorrelationData;
import org.springframework.amqp.support.AmqpHeaders;
import org.springframework.amqp.support.Correlation;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;

@Configuration
public class MORabbitMQConfiguration {

    @Value("${rabbitmq.connection.host}")
    private String hostname;
    @Value("${rabbitmq.connection.port}")
    private Integer port;
    @Value("${rabbitmq.connection.user}")
    private String userName;
    @Value("${rabbitmq.connection.password}")
    private String password;
    @Value("${rabbitmq.connection.virtual-host}")
    private String virtualHost;

    /* 15s/15s/30s/3m/10m */
    final int[] notifyFrequency = {15, 15, 30, 180, 600};

    @Resource
    private KoalaRabbitmqMessageMapperV1 koalaRabbitmqMessageMapperV1;

    @Bean
    public ConnectionFactory connectionFactory() {
        CachingConnectionFactory connectionFactory = new CachingConnectionFactory(hostname, port);
        connectionFactory.setUsername(userName);
        connectionFactory.setPassword(password);
        connectionFactory.setVirtualHost(virtualHost);
        connectionFactory.setPublisherConfirms(true);
        return connectionFactory;
    }

    @Bean
    public RabbitAdmin rabbitAdmin(ConnectionFactory connectionFactory) {
        return new RabbitAdmin(connectionFactory);
    }

    @Bean
    public RabbitTemplate moNotifyRabbitTemplate(ConnectionFactory connectionFactory) {
        RabbitTemplate rabbitTemplate = new RabbitTemplate(connectionFactory);
        rabbitTemplate.setConfirmCallback((correlationData, ack, cause) -> {
            if (ack) {
                koalaRabbitmqMessageMapperV1.updateMsgStatusByUniqueId(correlationData.getId(), Constants.MSG_STATUS_SUCCESS);
            }
        });
        rabbitTemplate.setBeforePublishPostProcessors(new CorrelationAwareMessagePostProcessor() {
            @Override
            public Message postProcessMessage(Message message) throws AmqpException {
                return message;
            }

            @Override
            public Message postProcessMessage(Message message, Correlation correlation) {
                String uniqueId = ((CorrelationData) correlation).getId();
                KoalaRabbitmqMessage koalaRabbitmqMessage = koalaRabbitmqMessageMapperV1.selectByUniqueId(uniqueId);
                if (koalaRabbitmqMessage == null) {
                    throw new RuntimeException("数据异常，消息不存在。");
                }
                if (koalaRabbitmqMessage.getConsumeCount() >= koalaRabbitmqMessage.getMaxConsumeCount()) {
                    throw new RuntimeException("消息id = 【" + uniqueId + "】消费已达最大消费次数");
                }
                message.getMessageProperties().setExpiration(String.valueOf(notifyFrequency[koalaRabbitmqMessage.getConsumeCount()] * 1000));
                message.getMessageProperties().getHeaders().put(AmqpHeaders.CORRELATION_ID, uniqueId);
                message.getMessageProperties().getHeaders().put("SCENARIO", "GRADIENT_NOTICE");
                return message;
            }
        });
        return rabbitTemplate;
    }

    @Bean
    public SimpleRabbitListenerContainerFactory simpleRabbitListenerContainerFactory(ConnectionFactory connectionFactory) {
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        factory.setAcknowledgeMode(AcknowledgeMode.MANUAL);
        return factory;
    }

    @Bean
    public MONotify moNotify() {
        return new MONotify();
    }
}
