package cn.quantgroup.boot.tech.brave.configuration;

import brave.Tracing;
import brave.context.slf4j.MDCScopeDecorator;
import brave.http.HttpTracing;
import brave.propagation.B3Propagation;
import brave.propagation.ExtraFieldPropagation;
import brave.propagation.ThreadLocalCurrentTraceContext;
import brave.spring.web.TracingClientHttpRequestInterceptor;
import brave.spring.webmvc.SpanCustomizingAsyncHandlerInterceptor;
import cn.quantgroup.boot.tech.brave.properties.BraveProperties;
import cn.quantgroup.boot.tech.brave.properties.ServiceProperties;
import cn.quantgroup.boot.tech.brave.service.*;
import cn.quantgroup.boot.tech.brave.service.impl.*;
import cn.quantgroup.tech.brave.service.*;
import cn.quantgroup.tech.brave.service.impl.*;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import org.apache.http.client.HttpClient;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import zipkin2.Span;
import zipkin2.codec.Encoding;
import zipkin2.reporter.AsyncReporter;
import zipkin2.reporter.Sender;
import zipkin2.reporter.kafka11.KafkaSender;

/**
 * 注册brave
 *
 * @author zhangheng
 * create on 2018.04.25
 */
@Slf4j
@Configuration
public class BraveAutoConfiguration {

    @Configuration
    public static class BraveUntEnabled {

        @Configuration
        @ConditionalOnClass(RabbitTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class RabbitTemplateEnable {
            /**
             * 提供RabbitTemplateBuilder,业务系统需要使用该Builder生成RabbitTemplate
             *
             * @return
             */
            @Bean(name = "techRabbitBuilder")
            public ITechRabbitBuilder techRabbitBuilder() {
                return new TechRabbitBuilderNoTrace();
            }
        }

        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechHttpClientEnable {
            /**
             * 提供TechHttpClientBuilder
             *
             * @return
             */
            @Bean(name = "techHttpClient")
            public ITechHttpClient techHttpClient() {
                log.info("注册ITechHttpClient, 没有开启brave");
                return new TechHttpClientNoTrace();
            }
        }

        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechOkHttpClientEnable {

            /**
             * 提供TechOkHttpClient
             *
             * @return
             */
            @Bean(name = "techOkHttpClient")
            public ITechOkHttpClient techOkHttpClient() {
                log.info("注册ITechOkClient, 没有开启brave");
                return new TechOkHttpClientNoTrace();
            }
        }

        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechRestTemplateEnable {
            /**
             * 提供TechRestTemplate
             *
             * @return
             */
            @Bean(name = "techRestTemplateBuilder")
            public ITechRestTemplateBuilder techRestTemplateBuilder() {
                log.info("注册TechRestTemplate, 没有开始brave");
                return new TechRestTemplateBuliderNoTrace();
            }
        }

        @Configuration
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechExecutorServiceEnable {

            @Bean(name = "techExecutorServiceBuilder")
            public ITechExecutorServiceBuilder techExecutorServiceBuilder() {
                return new TechExecutorServiceBuilderNoTrace();
            }
        }
    }

    @Configuration
    @EnableConfigurationProperties({BraveProperties.class, ServiceProperties.class})
    @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
    public static class BraveEnabled {
        @Autowired
        private BraveProperties braveProperties;

        @Autowired
        private ServiceProperties serviceProperties;


        @Bean
        @ConditionalOnMissingBean
        Sender sender() {
            return KafkaSender.newBuilder().bootstrapServers(braveProperties.getKafkaBootstrapServers()).topic(braveProperties.getKafkaTopic()).encoding(Encoding.JSON).build();
        }

        @Bean
        @ConditionalOnMissingBean
        AsyncReporter<Span> spanReporter() {
            return AsyncReporter.create(sender());
        }

        @Bean
        @ConditionalOnMissingBean
        Tracing tracing() {
            return Tracing.newBuilder()
                    .sampler(brave.sampler.Sampler.create(braveProperties.getSample()))
                    .localServiceName(serviceProperties.getName())
                    .propagationFactory(ExtraFieldPropagation.newFactory(B3Propagation.FACTORY, "user-name"))
                    .currentTraceContext(ThreadLocalCurrentTraceContext.newBuilder().addScopeDecorator(MDCScopeDecorator.get()).build())
                    .spanReporter(spanReporter()).build();
        }

        @Bean
        @ConditionalOnMissingBean
        HttpTracing httpTracing(Tracing tracing) {
            return HttpTracing.create(tracing);
        }

        @Configuration
        @ConditionalOnClass({WebMvcConfigurer.class})
        @Import(SpanCustomizingAsyncHandlerInterceptor.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class WebMvcEnable implements WebMvcConfigurer {

            @Autowired
            private SpanCustomizingAsyncHandlerInterceptor spanCustomizingAsyncHandlerInterceptor;


            @Override
            public void addInterceptors(InterceptorRegistry registry) {
                log.info("添加braveWebmvc拦截器");
                registry.addInterceptor(spanCustomizingAsyncHandlerInterceptor);
            }

        }

        @Configuration
        @ConditionalOnClass(RabbitTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class RabbitTemplateEnable {
            /**
             * 提供RabbitTemplateBuilder,业务系统需要使用该Builder生成RabbitTemplate
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techRabbitBuilder")
            public ITechRabbitBuilder rabbitTemplateBuilder(Tracing tracing) {
                return new TechRabbitBuilderTrace(tracing);
            }
        }

        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class TechHttpClientEnable {
            /**
             * 提供TechHttpClientBuilder
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techHttpClient")
            public ITechHttpClient techHttpClient(Tracing tracing) {
                log.info("注册ITechHttpClient, 已开启brave");
                return new TechHttpClientTrace(tracing);
            }
        }

        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class TechOkHttpClientEnable {
            /**
             * 提供TechOkHttpClient
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techOkHttpClient")
            public ITechOkHttpClient techOkHttpClient(Tracing tracing) {
                log.info("注册ITechOkHttpClient, 已开启brave");
                return new TechOkHttpClientTrace(tracing);
            }
        }


        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        @Import(TracingClientHttpRequestInterceptor.class)
        public static class TechRestTemplateEnable {
            /**
             * 提供RestTemplate
             *
             * @return
             */
            @Bean(name = "techRestTemplateBuilder")
            public ITechRestTemplateBuilder techRestTemplateBuilder() {
                return new TechRestTemplateBuilderTrace();
            }
        }

        @Configuration
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class TechExecutorServiceEnable {

            @Bean(name = "techExecutorServiceBuilder")
            public ITechExecutorServiceBuilder techExecutorServiceBuilder() {
                return new TechExecutorServiceBuilderTrace();
            }
        }
    }
}


