package cn.quantgroup.tech.util;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 注意事项：
 * 1. 使用前在配置文件内配置 DATA_CENTER_ID
 * <p>
 * <p>
 *
 * @author zhiguo.liu
 * @date 2017/5/18
 */
@Slf4j
@Component
@ConditionalOnBean(RedisTemplate.class)
@ConditionalOnProperty(name = "data.center.id")
public class IDGenerator {

    private static final String REDIS_WORK_ID_KEY = "GLOBAL:WORK:ID:";
    private static final String ID_FORMAT = "yyyyMMddHHmmss";
    /**
     * 最高支持同时1W台机器
     */
    private static final int MAX_WORK_ID = 10000;
    /**
     * 最高每秒发号 100w
     */
    private static final int MAX_COUNT = 999999;
    private static Lock lock = new ReentrantLock();
    /**
     * data center，默认为 1
     */
    private static int DATA_CENTER_ID = 1;
    private static AtomicInteger COUNTER = new AtomicInteger(0);
    private static long MAX_TIME_SECOND;

    /**
     * Worker ID 字符串
     */
    private static String WORKER_ID_STR;
    /**
     * data center 字符串
     */
    private static String DATA_CENTER_STR;
    /**
     * 最长回退时间，120 秒
     */
    private static int MAX_BACK_SECOND = 120;
    private static LoadingCache cache = CacheBuilder.newBuilder()
            .expireAfterWrite(MAX_BACK_SECOND, TimeUnit.SECONDS)
            .build(new CacheLoader<Long, AtomicInteger>() {
                @Override
                public AtomicInteger load(Long key) throws Exception {
                    return new AtomicInteger(0);
                }
            });

    static {
        Date now = new Date();
        MAX_TIME_SECOND = now.getTime() / 1000;
    }

    @Autowired
    private StringRedisTemplate redis;

    /**
     * 1. 需要获取 dataCenterId 和 workeId
     */
    public static String getId(String prefix) {
        Date now = new Date();
        Long timeSecond = now.getTime() / 1000;

        Integer counter = 0;
        if (timeSecond > MAX_TIME_SECOND) {
            lock.lock();
            if (timeSecond > MAX_TIME_SECOND) {
                cache.put(MAX_TIME_SECOND, COUNTER);
                COUNTER = new AtomicInteger(0);
                MAX_TIME_SECOND = timeSecond;
            }
            lock.unlock();
        }
        if (timeSecond == MAX_TIME_SECOND) {
            counter = COUNTER.incrementAndGet();
        }
        // 时间回退时到 cache 里拿，或者直接抛出错误
        if (timeSecond < MAX_TIME_SECOND) {
            if (timeSecond + MAX_BACK_SECOND < MAX_TIME_SECOND) {
                throw new RuntimeException("时间回撤, 请稍后再试");
            }
            try {
                AtomicInteger historyCounter = (AtomicInteger) cache.get(timeSecond);
                counter = historyCounter.incrementAndGet();
            } catch (ExecutionException e) {
                log.error("取出缓存时出错");
            }
        }

        // 达到计数器上上限, 休眠半秒并重试
        if (counter >= MAX_COUNT) {
            try {
                Thread.sleep(500);
                return getId(prefix);
            } catch (InterruptedException e) {
                log.error("发号器休眠时发生错误:{}", e);
            }
        }

        String currentTimeStr = new SimpleDateFormat(ID_FORMAT, Locale.SIMPLIFIED_CHINESE).format(now);
        return prefix + currentTimeStr + DATA_CENTER_STR + WORKER_ID_STR + counter;
    }

    @Value("${data.center.id}")
    public void setDataCenterId(Integer dataCenterId) {
        DATA_CENTER_ID = dataCenterId;
    }

    @PostConstruct
    public void init() {
        int workerId = (int) (redis.opsForValue().increment(REDIS_WORK_ID_KEY + DATA_CENTER_ID, 1) % MAX_WORK_ID);
        WORKER_ID_STR = String.valueOf(workerId);
        DATA_CENTER_STR = String.valueOf(DATA_CENTER_ID);
    }

}
