package cn.quantgroup.tech.shutdown.configuration;

import cn.quantgroup.tech.shutdown.properties.GracefulShutdownProperties;
import cn.quantgroup.tech.shutdown.service.JettyShutdown;
import cn.quantgroup.tech.shutdown.service.TomcatShutdown;
import cn.quantgroup.tech.shutdown.service.UndertowShutdown;
import cn.quantgroup.tech.shutdown.wrapper.UndertowShutdownHandlerWrapper;
import io.undertow.Undertow;
import org.apache.catalina.startup.Tomcat;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.util.Loader;
import org.eclipse.jetty.webapp.WebAppContext;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.boot.autoconfigure.web.EmbeddedServletContainerAutoConfiguration;
import org.springframework.boot.context.embedded.EmbeddedServletContainerCustomizer;
import org.springframework.boot.context.embedded.EmbeddedServletContainerFactory;
import org.springframework.boot.context.embedded.jetty.JettyEmbeddedServletContainerFactory;
import org.springframework.boot.context.embedded.tomcat.TomcatEmbeddedServletContainerFactory;
import org.springframework.boot.context.embedded.undertow.UndertowDeploymentInfoCustomizer;
import org.springframework.boot.context.embedded.undertow.UndertowEmbeddedServletContainerFactory;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import javax.servlet.Servlet;


/**
 * This configuration class will be picked up by Spring Boot's auto configuration capabilities as soon as it's
 * on the classpath.
 */
@Configuration
@ConditionalOnProperty(prefix = "shutdown.graceful", name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(GracefulShutdownProperties.class)
@Import(EmbeddedServletContainerAutoConfiguration.BeanPostProcessorsRegistrar.class)
public class GracefulShutdownAutoConfiguration {

    /**
     * Configuration for Tomcat.
     */
    @Configuration
    @ConditionalOnClass({Servlet.class, Tomcat.class})
    @ConditionalOnMissingBean(value = EmbeddedServletContainerFactory.class, search = SearchStrategy.CURRENT)
    public static class EmbeddedTomcat {

        @Bean
        public TomcatShutdown tomcatShutdown() {
            return new TomcatShutdown();
        }

        /**
         * Customise the tomcat factory.
         *
         * @return an EmbeddedServletContainerCustomizer
         */
        @Bean
        public EmbeddedServletContainerCustomizer tomcatCustomizer() {
            return container -> {
                if (container instanceof TomcatEmbeddedServletContainerFactory) {
                    ((TomcatEmbeddedServletContainerFactory) container).addConnectorCustomizers(tomcatShutdown());
                }
            };
        }
    }

    @Configuration
    @ConditionalOnClass({Servlet.class, Server.class, Loader.class,
            WebAppContext.class})
    @ConditionalOnMissingBean(value = EmbeddedServletContainerFactory.class, search = SearchStrategy.CURRENT)
    public static class EmbeddedJetty {


        @Bean
        public JettyShutdown jettyShutdown() {
            return new JettyShutdown();
        }


        @Bean
        public JettyEmbeddedServletContainerFactory jettyEmbeddedServletContainerFactory() {
            return new JettyEmbeddedServletContainerFactory();
        }

        @Bean
        public EmbeddedServletContainerCustomizer jettyCustomizer() {
            return container -> {
                if (container instanceof JettyEmbeddedServletContainerFactory) {
                    ((JettyEmbeddedServletContainerFactory) container).addServerCustomizers(jettyShutdown());
                }
            };
        }
    }


    /**
     * Configuration for Undertow.
     */
    @Configuration
    @ConditionalOnClass({Servlet.class, Undertow.class})
    @ConditionalOnMissingBean(value = EmbeddedServletContainerFactory.class, search = SearchStrategy.CURRENT)
    public static class EmbeddedUndertow {

        @Bean
        public UndertowShutdown undertowShutdown() {
            return new UndertowShutdown();
        }


        /**
         * Customise the undertow factory.
         *
         * @return an EmbeddedServletContainerCustomizer
         */
        @Bean
        public EmbeddedServletContainerCustomizer undertowCustomizer() {
            return container -> {
                if (container instanceof UndertowEmbeddedServletContainerFactory) {
                    ((UndertowEmbeddedServletContainerFactory) container).addDeploymentInfoCustomizers(undertowDeploymentInfoCustomizer());
                }
            };
        }

        @Bean
        public UndertowDeploymentInfoCustomizer undertowDeploymentInfoCustomizer() {
            return deploymentInfo -> deploymentInfo.addOuterHandlerChainWrapper(undertowShutdownHandlerWrapper());
        }

        @Bean
        public UndertowShutdownHandlerWrapper undertowShutdownHandlerWrapper() {
            return new UndertowShutdownHandlerWrapper();
        }
    }

}
