package cn.quantgroup.tech.generator;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.math.BigInteger;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * Represents an implementation of {@link IDGenerator}
 * 基于百度开源项目 uid-generator 的增强版，Snowflake Java 实现版本。项目 Github：https://github.com/baidu/uid-generator
 * <p>
 * uid-generator 通过对 64 位数字分区来生成唯一 ID，由以下组成：
 * <p>
 * -----------------------------------------------------------------------------------
 * | sign   |  delta seconds  |   worker id   |    SEQUENCE
 * -----------------------------------------------------------------------------------
 * 1bits         28bits          22bits           13bits
 * -----------------------------------------------------------------------------------
 * 其中 delta seconds 为 当前时间 - 指定起始时间。
 * 该版本有三个问题
 * 1. delta seconds 位数有限，28bits 也只能允许运行 8.7 年左右。
 * 2. worker id 生成号码为用后即弃，可容纳重启次数有限。
 * 3. 微服务分布式的情况下，无法使用统一数据源，则不同服务生成 worker id 时会重复
 * <p>
 * 于是做出以下改进
 * 1. worker id 拆分成 data center id，每个服务通过约定指定自己的 data center id 。
 * 2. worker id 通过 redis 自增指定，设计为首尾相连的环形，自增数字达到设定的最大值时，会从0开始。
 * 2. 不限制使用 delta seconds 的位数，则实现了无限时间的使用。当位数增长到 64 为后，改用 BigInteger 的位运算实现。
 * <p>
 * 经测试，BigInteger 实现时，性能降低 60% 左右，每秒发号约为 100w~150w。
 * 现在 uid 由以下组成
 * ---------------------------------------------------------------------------------------------------------
 * | sign(length < 64)   |  delta seconds (unlimited)  |    data center id   |   worker id   |    SEQUENCE
 * ---------------------------------------------------------------------------------------------------------
 * 1bits                         28bits                      22bits             22bits           13bits
 * ---------------------------------------------------------------------------------------------------------
 * 其中 data center id + worker id + SEQUENCE 设定的位数不大于 63。
 * <p>
 * 使用注意：
 * 1. 号码的位数不固定，会随着时间增长。data center id + worker id + SEQUENCE 总数设定越大，号码位数越长
 * 2. 各个分区的位数、起始时间一旦设定完成投入使用，则后续不能更改。否则会导致发号重复。
 *
 * @author zhiguo.liu
 */
@Slf4j
public class IDGenerator {
    private static final String DATETIME_PATTERN = "yyyy-MM-dd HH:mm:ss";


    private long dataCenterId;
    /**
     * Customer epoch, unit as second. For example 2018-03-01 (ms: 1463673600000)
     */
    private long epochSeconds;
    /**
     * Stable fields after spring bean initializing
     */
    private BitsAllocator allocator;
    private long workerId;
    /**
     * Volatile fields caused by nextId()
     */
    private long sequence = 0L;
    private long lastSecond = -1L;


    public IDGenerator(Long dataCenterId, Long epochSeconds, Long workerId, BitsAllocator allocator) {
        this.dataCenterId = dataCenterId;
        this.epochSeconds = epochSeconds;
        this.workerId = workerId;
        this.allocator = allocator;
    }


    public String getID(String prefix) throws IDGenerateException {
        try {
            return nextId(prefix);
        } catch (Exception e) {
            log.error("Generate unique id exception. ", e);
            throw new IDGenerateException(e);
        }
    }

    public String parseID(String idStr) {
        BigInteger bigInteger = new BigInteger(idStr);

        int totalBits = bigInteger.bitLength();
        long dataCenterIdBits = allocator.getDataCenterIdBits();
        long workerIdBits = allocator.getWorkerIdBits();
        long sequenceBits = allocator.getSequenceBits();
        if (totalBits < 64) {
            totalBits = 64;
            long id = bigInteger.longValue();
            long sequence = (id << (totalBits - sequenceBits)) >>> (totalBits - sequenceBits);
            long workerId = (id << (totalBits - workerIdBits - sequenceBits)) >>> (totalBits - workerIdBits);
            long dataCenterId = (id << (totalBits - dataCenterIdBits - workerIdBits - sequenceBits)) >>> (totalBits - dataCenterIdBits);
            if (dataCenterIdBits == 0) {
                dataCenterId = 0;
            }
            long deltaSeconds = id >>> (dataCenterIdBits + workerIdBits + sequenceBits);
            Date thatTime = new Date(TimeUnit.SECONDS.toMillis(epochSeconds + deltaSeconds));
            String thatTimeStr = DateFormatUtils.format(thatTime, DATETIME_PATTERN);
            return String.format("{\"ID\":\"%d\",\"timestamp\":\"%s\",\"DATA_CENTER_ID\":\"%d\",\"WORKER_ID\":\"%d\",\"SEQUENCE\":\"%d\"}",
                    id, thatTimeStr, dataCenterId, workerId, sequence);
        } else {
            long sequence = getBigIntegerFromLength(sequenceBits).and(bigInteger).longValue();
            long workerId = getBigIntegerFromLength(workerIdBits).and(bigInteger.shiftRight((int) sequenceBits)).longValue();
            long dataCenterId = getBigIntegerFromLength(dataCenterIdBits).and(bigInteger.shiftRight((int) sequenceBits + (int) workerIdBits)).longValue();
            if (dataCenterIdBits == 0) {
                dataCenterId = 0;
            }
            long deltaSeconds = bigInteger.shiftRight((int) dataCenterIdBits + (int) workerIdBits + (int) sequenceBits).longValue();
            Date thatTime = new Date(TimeUnit.SECONDS.toMillis(epochSeconds + deltaSeconds));
            String thatTimeStr = DateFormatUtils.format(thatTime, DATETIME_PATTERN);
            return String.format("{\"ID\":\"%d\",\"timestamp\":\"%s\",\"DATA_CENTER_ID\":\"%d\",\"WORKER_ID\":\"%d\",\"SEQUENCE\":\"%d\"}",
                    bigInteger, thatTimeStr, dataCenterId, workerId, sequence);
        }
    }

    private BigInteger getBigIntegerFromLength(long n) {
        return BigInteger.valueOf(-1).shiftLeft((int) n).not();
    }

    /**
     * Get ID
     *
     * @return ID
     * @throws IDGenerateException in the case: Clock moved backwards; Exceeds the max timestamp
     */
    protected synchronized String nextId(String preFix) {
        long currentSecond = getCurrentSecond();

        // Clock moved backwards, wait for newest time
        if (currentSecond < lastSecond) {
            getNextSecond(lastSecond);
        }

        // At the same second, increase SEQUENCE
        if (currentSecond == lastSecond) {
            sequence = (sequence + 1) & allocator.getMaxSequence();
            // Exceed the max SEQUENCE, we wait the next second to generate ID
            if (sequence == 0) {
                currentSecond = getNextSecond(lastSecond);
            }
            // At the different second, SEQUENCE restart from zero
        } else {
            sequence = 0L;
        }

        lastSecond = currentSecond;

        // 当前时间小于设定的最大时间，即总位数在 64 位以下，用 long 生成数字
        if (currentSecond - epochSeconds <= allocator.getMaxDeltaSeconds()) {
            return preFix + allocator.allocate(currentSecond - epochSeconds, dataCenterId, workerId, sequence);
        }
        return preFix + allocator.allocateBigInteger(currentSecond - epochSeconds, dataCenterId, workerId, sequence);
    }

    /**
     * Get next millisecond
     */
    private long getNextSecond(long lastTimestamp) {
        long timestamp = getCurrentSecond();
        while (timestamp <= lastTimestamp) {
            timestamp = getCurrentSecond();
        }

        return timestamp;
    }

    /**
     * Get current second
     */
    private long getCurrentSecond() {
        return TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
    }

}
