package cn.quantgroup.tech.shutdown.service;

import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.Connector;
import org.apache.tomcat.util.threads.ThreadPoolExecutor;
import org.springframework.boot.context.embedded.tomcat.TomcatConnectorCustomizer;

import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

/**
 * Perform a tomcat shutdown.
 */
@Slf4j
public class TomcatShutdown implements Shutdown, TomcatConnectorCustomizer {

    /**
     * Implementation of a Coyote connector.
     */
    private volatile Connector connector;

    /**
     * Perform a shutdown
     *
     * @param delay is delay to force is the delay before perform a force shutdown
     * @throws InterruptedException if we have an exception
     */
    @Override
    public void shutdown(Integer delay) throws InterruptedException {
        // Used to properly handle the work queue.
        final Executor executor = connector.getProtocolHandler().getExecutor();
        final ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor) executor;
        /*
         * Initiates an orderly shutdown in which previously submitted
         * tasks are executed, but no new tasks will be accepted.
         * Invocation has no additional effect if already shut down.
         */
        threadPoolExecutor.shutdown();

        // We wait after the end of the current requests
        if (!threadPoolExecutor.awaitTermination(delay, TimeUnit.SECONDS)) {
            log.warn("Tomcat thread pool did not shut down gracefully within " + delay + " second(s). Proceeding with force shutdown");
        } else {
            log.debug("Tomcat thread pool is empty, we stop now");
        }
    }

    /**
     * Set connector.
     *
     * @param connector is the catalina connector.
     */
    @Override
    public void customize(final Connector connector) {
        this.connector = connector;
    }
}
