package cn.quantgroup.tech.brave.configuration;

import brave.Tracing;
import brave.context.slf4j.MDCScopeDecorator;
import brave.http.HttpTracing;
import brave.httpclient.TracingHttpClientBuilder;
import brave.okhttp3.TracingInterceptor;
import brave.propagation.B3Propagation;
import brave.propagation.ExtraFieldPropagation;
import brave.propagation.ThreadLocalCurrentTraceContext;
import brave.spring.web.TracingClientHttpRequestInterceptor;
import brave.spring.webmvc.SpanCustomizingAsyncHandlerInterceptor;
import cn.quantgroup.tech.brave.filter.CustomDelegatingTracingFilter;
import cn.quantgroup.tech.brave.properties.BraveProperties;
import cn.quantgroup.tech.brave.properties.ServiceProperties;
import cn.quantgroup.tech.brave.service.*;
import cn.quantgroup.tech.brave.service.impl.*;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import zipkin2.Span;
import zipkin2.codec.Encoding;
import zipkin2.reporter.AsyncReporter;
import zipkin2.reporter.Sender;
import zipkin2.reporter.kafka11.KafkaSender;

import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.List;

/**
 * 注册brave
 *
 * @author zhangheng
 * create on 2018.04.25
 */
@Slf4j
@Configuration
public class BraveAutoConfiguration {

    @Configuration
    public static class BraveUntEnabled {

        @Configuration
        @ConditionalOnClass(RabbitTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class RabbitTemplateEnable {
            /**
             * 提供RabbitTemplateBuilder,业务系统需要使用该Builder生成RabbitTemplate
             *
             * @return
             */
            @Bean(name = "techRabbitBuilder")
            public ITechRabbitBuilder techRabbitBuilder() {
                return new TechRabbitBuilderNoTrace();
            }
        }

        @Deprecated
        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class HttpClientEnable {
            /**
             * 提供HttpClientBuilderBean
             *
             * @return
             * @see ITechHttpClient
             */
            @Deprecated
            @Bean(name = "techHttpClientBuilder")
            public HttpClientBuilder httpClientBuilder() {
                log.info("注册HttpClientBuilder");
                return HttpClientBuilder.create();
            }
        }

        @Deprecated
        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class OkHttpClientEnable {

            /**
             * 提供OkHttpClientBean
             *
             * @return
             * @see ITechOkHttpClient
             */
            @Deprecated
            @Bean(name = "techOkHttpClientBuilder")
            public OkHttpClient.Builder okHttpClientBuilder() {
                log.info("注册OkHttpClient.Builder");
                return new OkHttpClient.Builder();
            }
        }

        @Deprecated
        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class RestTemplateEnable {
            /**
             * 提供RestTemplate
             *
             * @return
             * @see ITechRestTemplateBuilder
             */
            @Deprecated
            @Bean(name = "techRestTemplate")
            public RestTemplate restTemplate() {
                log.info("添加restTemplate拦截器");
                return new RestTemplate();
            }
        }

        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechHttpClientEnable {
            /**
             * 提供TechHttpClientBuilder
             *
             * @return
             */
            @Bean(name = "techHttpClient")
            public ITechHttpClient techHttpClient() {
                log.info("注册ITechHttpClient, 没有开启brave");
                return new TechHttpClientNoTrace();
            }
        }

        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechOkHttpClientEnable {

            /**
             * 提供TechOkHttpClient
             *
             * @return
             */
            @Bean(name = "techOkHttpClient")
            public ITechOkHttpClient techOkHttpClient() {
                log.info("注册ITechOkClient, 没有开启brave");
                return new TechOkHttpClientNoTrace();
            }
        }

        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechRestTemplateEnable {
            /**
             * 提供TechRestTemplate
             *
             * @return
             */
            @Bean(name = "techRestTemplateBuilder")
            public ITechRestTemplateBuilder techRestTemplateBuilder() {
                log.info("注册TechRestTemplate, 没有开始brave");
                return new TechRestTemplateBuliderNoTrace();
            }
        }

        @Configuration
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class TechExecutorServiceEnable {

            @Bean(name = "techExecutorServiceBuilder")
            public ITechExecutorServiceBuilder techExecutorServiceBuilder() {
                return new TechExecutorServiceBuilderNoTrace();
            }
        }
    }

    @Configuration
    @EnableConfigurationProperties({BraveProperties.class, ServiceProperties.class})
    @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
    public static class BraveEnabled {
        @Autowired
        private BraveProperties braveProperties;

        @Autowired
        private ServiceProperties serviceProperties;


        @Bean
        @ConditionalOnMissingBean
        Sender sender() {
            return KafkaSender.newBuilder().bootstrapServers(braveProperties.getKafkaBootstrapServers()).topic(braveProperties.getKafkaTopic()).encoding(Encoding.JSON).build();
        }

        @Bean
        @ConditionalOnMissingBean
        AsyncReporter<Span> spanReporter() {
            return AsyncReporter.create(sender());
        }

        @Bean
        @ConditionalOnMissingBean
        Tracing tracing() {
            return Tracing.newBuilder()
                    .sampler(brave.sampler.Sampler.create(braveProperties.getSample()))
                    .localServiceName(serviceProperties.getName())
                    .propagationFactory(ExtraFieldPropagation.newFactory(B3Propagation.FACTORY, "user-name"))
                    .currentTraceContext(ThreadLocalCurrentTraceContext.newBuilder().addScopeDecorator(MDCScopeDecorator.get()).build())
                    .spanReporter(spanReporter()).build();
        }

        @Deprecated
        @Bean
        @ConditionalOnMissingBean
        public HttpTracing httpTracing(Tracing tracing) {
            return HttpTracing.create(tracing);
        }

        @Configuration
        @ConditionalOnWebApplication
        @ConditionalOnClass({Filter.class})
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class ServletEnable {

            @Bean
            public FilterRegistrationBean registrationBean() {
                FilterRegistrationBean registrationBean = new FilterRegistrationBean();
                CustomDelegatingTracingFilter delegatingTracingFilter = new CustomDelegatingTracingFilter();
                registrationBean.setFilter(delegatingTracingFilter);
                registrationBean.setName("techDelegatingTracingFilter");
                registrationBean.setOrder(Integer.MIN_VALUE);
                registrationBean.addUrlPatterns("/*");
                return registrationBean;
            }
        }

        @Configuration
        @ConditionalOnClass({WebMvcConfigurerAdapter.class})
        @Import(SpanCustomizingAsyncHandlerInterceptor.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class WebMvcEnable extends WebMvcConfigurerAdapter {

            @Autowired
            private SpanCustomizingAsyncHandlerInterceptor spanCustomizingAsyncHandlerInterceptor;


            @Override
            public void addInterceptors(InterceptorRegistry registry) {
                log.info("添加braveWebmvc拦截器");
                registry.addInterceptor(spanCustomizingAsyncHandlerInterceptor);
            }

        }

        @Configuration
        @ConditionalOnClass(RabbitTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class RabbitTemplateEnable {
            /**
             * 提供RabbitTemplateBuilder,业务系统需要使用该Builder生成RabbitTemplate
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techRabbitBuilder")
            public ITechRabbitBuilder rabbitTemplateBuilder(Tracing tracing) {
                return new TechRabbitBuilderTrace(tracing);
            }
        }

        @Deprecated
        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class HttpClientEnable {
            /**
             * 提供HttpClientBuilderBean
             *
             * @param tracing
             * @return
             * @see ITechHttpClient
             */
            @Deprecated
            @Bean(name = "techHttpClientBuilder")
            public HttpClientBuilder httpClientBuilder(Tracing tracing) {
                log.info("注册braveHttpClientBuilder");
                return TracingHttpClientBuilder.create(tracing);
            }
        }

        @Deprecated
        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class OkHttpClientEnable {
            /**
             * 提供OkHttpClientBean
             *
             * @param httpTracing
             * @return
             * @see ITechOkHttpClient
             */
            @Deprecated
            @Bean(name = "techOkHttpClientBuilder")
            public OkHttpClient.Builder okHttpClientBuilder(HttpTracing httpTracing) {
                log.info("注册braveOkHttpClient.Builder");
                return new OkHttpClient.Builder()
                        .dispatcher(new Dispatcher(httpTracing.tracing().currentTraceContext().executorService(new Dispatcher().executorService())))
                        .addNetworkInterceptor(TracingInterceptor.create(httpTracing));
            }
        }


        @Deprecated
        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        @Import(TracingClientHttpRequestInterceptor.class)
        public static class RestTemplateEnable {
            @Autowired
            private TracingClientHttpRequestInterceptor clientInterceptor;

            /**
             * 提供RestTemplate
             *
             * @return
             * @see ITechRestTemplateBuilder
             */
            @Deprecated
            @Bean(name = "techRestTemplate")
            public RestTemplate restTemplate() {
                log.info("添加braveRestTemplate拦截器");
                RestTemplate restTemplate = new RestTemplate();
                List<ClientHttpRequestInterceptor> interceptors =
                        new ArrayList<>(restTemplate.getInterceptors());
                interceptors.add(clientInterceptor);
                restTemplate.setInterceptors(interceptors);
                return restTemplate;
            }
        }

        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class TechHttpClientEnable {
            /**
             * 提供TechHttpClientBuilder
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techHttpClient")
            public ITechHttpClient techHttpClient(Tracing tracing) {
                log.info("注册ITechHttpClient, 已开启brave");
                return new TechHttpClientTrace(tracing);
            }
        }

        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class TechOkHttpClientEnable {
            /**
             * 提供TechOkHttpClient
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techOkHttpClient")
            public ITechOkHttpClient techOkHttpClient(Tracing tracing) {
                log.info("注册ITechOkHttpClient, 已开启brave");
                return new TechOkHttpClientTrace(tracing);
            }
        }


        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        @Import(TracingClientHttpRequestInterceptor.class)
        public static class TechRestTemplateEnable {
            /**
             * 提供RestTemplate
             *
             * @return
             */
            @Bean(name = "techRestTemplateBuilder")
            public ITechRestTemplateBuilder techRestTemplateBuilder() {
                return new TechRestTemplateBuilderTrace();
            }
        }

        @Configuration
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class TechExecutorServiceEnable {

            @Bean(name = "techExecutorServiceBuilder")
            public ITechExecutorServiceBuilder techExecutorServiceBuilder() {
                return new TechExecutorServiceBuilderTrace();
            }
        }
    }
}


