package cn.quantgroup.tech.brave.configuration;

import brave.Tracing;
import brave.context.slf4j.MDCCurrentTraceContext;
import brave.http.HttpTracing;
import brave.httpclient.TracingHttpClientBuilder;
import brave.okhttp3.TracingInterceptor;
import brave.propagation.B3Propagation;
import brave.propagation.ExtraFieldPropagation;
import brave.spring.web.TracingClientHttpRequestInterceptor;
import brave.spring.webmvc.DelegatingTracingFilter;
import brave.spring.webmvc.SpanCustomizingAsyncHandlerInterceptor;
import cn.quantgroup.tech.brave.properties.BraveProperties;
import cn.quantgroup.tech.brave.properties.ServiceProperties;
import cn.quantgroup.tech.brave.service.ITechRabbitBuilder;
import cn.quantgroup.tech.brave.service.TechRabbitBuilderNoTrace;
import cn.quantgroup.tech.brave.service.TechRabbitBuilderTrace;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import zipkin2.Span;
import zipkin2.codec.Encoding;
import zipkin2.reporter.AsyncReporter;
import zipkin2.reporter.Sender;
import zipkin2.reporter.kafka11.KafkaSender;

import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.List;

/**
 * 注册brave
 *
 * @author zhangheng
 * create on 2018.04.25
 */
@Slf4j
@Configuration
public class BraveAutoConfiguration {

    @Configuration
    public static class BraveUntEnabled {
        @Configuration
        @ConditionalOnClass(RabbitTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class RabbitTemplateEnable {
            /**
             * 提供RabbitTemplateBuilder,业务系统需要使用该Builder生成RabbitTemplate
             *
             * @return
             */
            @Bean(name = "techRabbitBuilder")
            public ITechRabbitBuilder techRabbitBuilder() {
                return new TechRabbitBuilderNoTrace();
            }
        }

        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class HttpClientEnable {
            /**
             * 提供HttpClientBuilderBean
             *
             * @return
             */
            @Bean(name = "techHttpClientBuilder")
            public HttpClientBuilder httpClientBuilder() {
                log.info("注册HttpClientBuilder");
                return HttpClientBuilder.create();
            }
        }

        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class OkHttpClientEnable {

            /**
             * 提供OkHttpClientBean
             *
             * @return
             */
            @Bean(name = "techOkHttpClientBuilder")
            public OkHttpClient.Builder okHttpClientBuilder() {
                log.info("注册OkHttpClient.Builder");
                return new OkHttpClient.Builder();
            }
        }

        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "false", matchIfMissing = true)
        public static class RestTemplateEnable {
            /**
             * 提供RestTemplate
             *
             * @return
             */
            @Bean(name = "techRestTemplate")
            public RestTemplate restTemplate() {
                log.info("添加restTemplate拦截器");
                return new RestTemplate();
            }
        }

    }

    @Configuration
    @EnableConfigurationProperties({BraveProperties.class, ServiceProperties.class})
    @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
    public static class BraveEnabled {
        @Autowired
        private BraveProperties braveProperties;

        @Autowired
        private ServiceProperties serviceProperties;


        @Bean
        @ConditionalOnMissingBean
        Sender sender() {
            return KafkaSender.newBuilder().bootstrapServers(braveProperties.getKafkaBootstrapServers()).topic(braveProperties.getKafkaTopic()).encoding(Encoding.JSON).build();
        }

        @Bean
        @ConditionalOnMissingBean
        AsyncReporter<Span> spanReporter() {
            return AsyncReporter.create(sender());
        }

        @Bean
        @ConditionalOnMissingBean
        Tracing tracing() {
            return Tracing.newBuilder()
                    .sampler(brave.sampler.Sampler.create(braveProperties.getSample()))
                    .localServiceName(serviceProperties.getName())
                    .propagationFactory(ExtraFieldPropagation.newFactory(B3Propagation.FACTORY, "user-name"))
                    .currentTraceContext(MDCCurrentTraceContext.create())
                    .spanReporter(spanReporter()).build();
        }

        @Bean
        @ConditionalOnMissingBean
        public HttpTracing httpTracing(Tracing tracing) {
            return HttpTracing.create(tracing);
        }


        @Configuration
        @ConditionalOnWebApplication
        @ConditionalOnClass({Filter.class})
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class ServletEnable {
            /**
             * 创建一个bean
             *
             * @return
             */
            @Bean(name = "techDelegatingTracingFilter")
            public Filter delegatingTracingFilter() {
                log.info("注册DelegatingTracingFilter");
                return new DelegatingTracingFilter();
            }

        }

        @Configuration
        @ConditionalOnClass({WebMvcConfigurerAdapter.class})
        @Import(SpanCustomizingAsyncHandlerInterceptor.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class WebMvcEnable extends WebMvcConfigurerAdapter {

            @Autowired
            private SpanCustomizingAsyncHandlerInterceptor spanCustomizingAsyncHandlerInterceptor;


            @Override
            public void addInterceptors(InterceptorRegistry registry) {
                log.info("添加braveWebmvc拦截器");
                registry.addInterceptor(spanCustomizingAsyncHandlerInterceptor);
            }

        }

        @Configuration
        @ConditionalOnClass(RabbitTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class RabbitTemplateEnable {
            /**
             * 提供RabbitTemplateBuilder,业务系统需要使用该Builder生成RabbitTemplate
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techRabbitBuilder")
            public ITechRabbitBuilder rabbitTemplateBuilder(Tracing tracing) {
                return new TechRabbitBuilderTrace(tracing);
            }
        }

        @Configuration
        @ConditionalOnClass(HttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class HttpClientEnable {
            /**
             * 提供HttpClientBuilderBean
             *
             * @param tracing
             * @return
             */
            @Bean(name = "techHttpClientBuilder")
            public HttpClientBuilder httpClientBuilder(Tracing tracing) {
                log.info("注册braveHttpClientBuilder");
                return TracingHttpClientBuilder.create(tracing);
            }
        }

        @Configuration
        @ConditionalOnClass(OkHttpClient.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        public static class OkHttpClientEnable {
            /**
             * 提供OkHttpClientBean
             *
             * @param httpTracing
             * @return
             */
            @Bean(name = "techOkHttpClientBuilder")
            public OkHttpClient.Builder okHttpClientBuilder(HttpTracing httpTracing) {
                log.info("注册braveOkHttpClient.Builder");
                return new OkHttpClient.Builder()
                        .dispatcher(new Dispatcher(httpTracing.tracing().currentTraceContext().executorService(new Dispatcher().executorService())))
                        .addNetworkInterceptor(TracingInterceptor.create(httpTracing));
            }
        }


        @Configuration
        @ConditionalOnClass(RestTemplate.class)
        @ConditionalOnProperty(prefix = "tech.brave", name = "enabled", havingValue = "true", matchIfMissing = true)
        @Import(TracingClientHttpRequestInterceptor.class)
        public static class RestTemplateEnable {
            @Autowired
            private TracingClientHttpRequestInterceptor clientInterceptor;

            /**
             * 提供RestTemplate
             *
             * @return
             */
            @Bean(name = "techRestTemplate")
            public RestTemplate restTemplate() {
                log.info("添加braveRestTemplate拦截器");
                RestTemplate restTemplate = new RestTemplate();
                List<ClientHttpRequestInterceptor> interceptors =
                        new ArrayList<>(restTemplate.getInterceptors());
                interceptors.add(clientInterceptor);
                restTemplate.setInterceptors(interceptors);
                return restTemplate;
            }
        }

    }

}


