package cn.quantgroup.tech.util.id;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.math.BigInteger;
import java.text.ParseException;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * Represents an implementation of {@link UidGenerator}
 * <p>
 * The unique id has 64bits (long), default allocated as blow:<br>
 * <li>sign: The highest bit is 0
 * <li>delta seconds: The next 28 bits, represents delta seconds since a customer epoch(2016-05-20 00:00:00.000).
 * Supports about 8.7 years until to 2024-11-20 21:24:16
 * <li>worker id: The next 22 bits, represents the worker's id which assigns based on database, max id is about 420W
 * <li>sequence: The next 13 bits, represents a sequence within the same second, max for 8192/s<br><br>
 * <p>
 * <p>
 * <pre>{@code
 * +------+----------------------+----------------+-----------+
 * | sign |     delta seconds    | worker node id | sequence  |
 * +------+----------------------+----------------+-----------+
 *   1bit          28bits              22bits         13bits
 * }</pre>
 * <p>
 * You can also specified the bits by Spring property setting.
 * <li>timeBits: default as 28
 * <li>workerBits: default as 22
 * <li>seqBits: default as 13
 * <li>epochStr: Epoch date string format 'yyyy-MM-dd'. Default as '2016-05-20'<p>
 * <p>
 * <b>Note that:</b> The total bits must be 64 -1
 *
 * @author yutianbao
 */
@Component
public class DefaultUidGenerator implements UidGenerator, InitializingBean {
    public static final String DATETIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static final String DAY_PATTERN = "yyyy-MM-dd";
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultUidGenerator.class);
    /**
     * Bits allocate
     */
    @Value("${uid.timeBits:28}")
    protected int timeBits;
    @Value("${uid.dataCenterIdBits:9}")
    protected int dataCenterIdBits;
    @Value("${uid.workerBits:13}")
    protected int workerBits;
    @Value("${uid.seqBits:13}")
    protected int seqBits;

    /**
     * Customer epoch, unit as second. For example 2018-03-01 (ms: 1463673600000)
     */
    protected String epochStr;
    protected long epochSeconds;

    /**
     * Stable fields after spring bean initializing
     */
    protected BitsAllocator bitsAllocator;
    protected long workerId;
    protected long dataCenterId;

    /**
     * Volatile fields caused by nextId()
     */
    protected long sequence = 0L;
    protected long lastSecond = -1L;

    /**
     * Spring property
     */
    @Autowired
    protected DisposableWorkerIdAssigner workerIdAssigner;

    @Override
    public void afterPropertiesSet() throws Exception {
        // initialize bits allocator
        bitsAllocator = new BitsAllocator(timeBits, dataCenterIdBits, workerBits, seqBits);
        workerIdAssigner = new DisposableWorkerIdAssigner();
        // initialize worker id
        workerId = workerIdAssigner.assignWorkerId(dataCenterId, bitsAllocator);
        Assert.isTrue(workerId < bitsAllocator.getMaxWorkerId(), "workerId is too big");
        Assert.isTrue(dataCenterId < bitsAllocator.getMaxDataCenterId(), "dataCenterId is too big");
        LOGGER.info("Initialized bits dataCenterBits:{}, workerBits:{}, seqBits:{}", dataCenterIdBits, workerBits, seqBits);
        LOGGER.info("Initialized nodes, workerId:{}, dataCenterId:{}", workerId, dataCenterId);
    }

    @Override
    public String getUID(String preFix) throws UidGenerateException {
        try {
            return nextId(preFix);
        } catch (Exception e) {
            LOGGER.error("Generate unique id exception. ", e);
            throw new UidGenerateException(e);
        }
    }

    // TODO: 2018/3/5 反序列化 uid
    @Override
    public String parseUID(String uidStr) {
        BigInteger bigInteger = new BigInteger(uidStr);

        int totalBits = bigInteger.bitLength();
        long dataCenterIdBits = bitsAllocator.getDataCenterIdBits();
        long workerIdBits = bitsAllocator.getWorkerIdBits();
        long sequenceBits = bitsAllocator.getSequenceBits();
        if (totalBits < 64) {
            totalBits = 64;
            long uid = bigInteger.longValue();
            long sequence = (uid << (totalBits - sequenceBits)) >>> (totalBits - sequenceBits);
            long workerId = (uid << (totalBits - workerIdBits - sequenceBits)) >>> (totalBits - workerIdBits);
            long dataCenterId = (uid << (totalBits - dataCenterIdBits - workerIdBits - sequenceBits)) >>> (totalBits - dataCenterIdBits);
            long deltaSeconds = uid >>> (dataCenterIdBits + workerIdBits + sequenceBits);
            Date thatTime = new Date(TimeUnit.SECONDS.toMillis(epochSeconds + deltaSeconds));
            String thatTimeStr = DateFormatUtils.format(thatTime, DATETIME_PATTERN);
            return String.format("{\"UID\":\"%d\",\"timestamp\":\"%s\",\"dataCenterId\":\"%d\",\"workerId\":\"%d\",\"sequence\":\"%d\"}",
                    uid, thatTimeStr, dataCenterId, workerId, sequence);
        } else {
            BigInteger workerBig = getBigIntegerFromLength(workerIdBits).shiftLeft((int) sequenceBits).and(bigInteger);
            System.out.println(workerBig);
            long sequence = getBigIntegerFromLength(sequenceBits).and(bigInteger).longValue();
            long workerId = getBigIntegerFromLength(workerIdBits).and(bigInteger.shiftRight((int)sequenceBits)).longValue();
            long dataCenterId = getBigIntegerFromLength(dataCenterIdBits).and(bigInteger.shiftRight((int)sequenceBits+(int)workerIdBits)).longValue();
            long deltaSeconds = bigInteger.shiftRight((int) dataCenterIdBits + (int) workerIdBits + (int) sequenceBits).longValue();
            Date thatTime = new Date(TimeUnit.SECONDS.toMillis(epochSeconds + deltaSeconds));
            String thatTimeStr = DateFormatUtils.format(thatTime, DATETIME_PATTERN);
            return String.format("{\"UID\":\"%d\",\"timestamp\":\"%s\",\"dataCenterId\":\"%d\",\"workerId\":\"%d\",\"sequence\":\"%d\"}",
                    bigInteger, thatTimeStr, dataCenterId, workerId, sequence);
        }
    }

    private BigInteger getBigIntegerFromLength(long n) {
        return BigInteger.valueOf(-1).shiftLeft((int) n).not();
    }

    /**
     * Get UID
     *
     * @return UID
     * @throws UidGenerateException in the case: Clock moved backwards; Exceeds the max timestamp
     */
    protected synchronized String nextId(String preFix) {
        long currentSecond = getCurrentSecond();

        // Clock moved backwards, wait for newest time
        if (currentSecond < lastSecond) {
            getNextSecond(lastSecond);
        }

        // At the same second, increase sequence
        if (currentSecond == lastSecond) {
            sequence = (sequence + 1) & bitsAllocator.getMaxSequence();
            // Exceed the max sequence, we wait the next second to generate uid
            if (sequence == 0) {
                currentSecond = getNextSecond(lastSecond);
            }
            // At the different second, sequence restart from zero
        } else {
            sequence = 0L;
        }

        lastSecond = currentSecond;

        // 当前时间小于设定的最大时间，即总位数在 64 位以下，用 long 生成数字
        if (currentSecond - epochSeconds <= bitsAllocator.getMaxDeltaSeconds()) {
            return preFix + bitsAllocator.allocate(currentSecond - epochSeconds, dataCenterId, workerId, sequence);
        }
        return preFix + bitsAllocator.allocateBigInteger(currentSecond - epochSeconds, dataCenterId, workerId, sequence);
    }

    /**
     * Get next millisecond
     */
    private long getNextSecond(long lastTimestamp) {
        long timestamp = getCurrentSecond();
        while (timestamp <= lastTimestamp) {
            timestamp = getCurrentSecond();
        }

        return timestamp;
    }

    /**
     * Get current second
     */
    private long getCurrentSecond() {
        return TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
    }

    public void setTimeBits(int timeBits) {
        if (timeBits > 0) {
            this.timeBits = timeBits;
        }
    }

    public void setWorkerBits(int workerBits) {
        if (workerBits > 0) {
            this.workerBits = workerBits;
        }
    }

    public void setSeqBits(int seqBits) {
        if (seqBits > 0) {
            this.seqBits = seqBits;
        }
    }

    @Value("${uid.epochStr:2018-03-01}")
    public void setEpochStr(String epochStr) {
        if (StringUtils.isNotBlank(epochStr)) {
            this.epochStr = epochStr;
            try {
                this.epochSeconds = TimeUnit.MILLISECONDS.toSeconds(DateUtils.parseDate(epochStr, new String[]{DAY_PATTERN}).getTime());
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
    }
}
