package cn.quantgroup.boot.tech.generator.configuration;

import cn.quantgroup.boot.tech.generator.BitsAllocator;
import cn.quantgroup.boot.tech.generator.IDGenerator;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.util.Assert;

import java.util.concurrent.TimeUnit;


@Slf4j
@Builder
public class GeneratorFactoryBean implements FactoryBean<IDGenerator>, InitializingBean {


    private static final String REDIS_WORK_ID_KEY = "GLOBAL:WORK:ID:";
    private static final String DAY_PATTERN = "yyyy-MM-dd";

    /**
     * from config
     */
    private int dataCenterIdBits;
    private int seqBits;
    private int workerBits;
    private long dataCenterId;
    private String epochStr;
    private long workerId;
    private StringRedisTemplate stringRedisTemplate;

    /**
     * local construct
     */
    private long epochSeconds;
    private BitsAllocator allocator;
    private IDGenerator idGenerator;


    @Override
    public IDGenerator getObject() throws Exception {
        return idGenerator;
    }

    @Override
    public Class<?> getObjectType() {
        return IDGenerator.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    @Override
    public void afterPropertiesSet() throws Exception {

        if (stringRedisTemplate == null) {
            log.error("redis template is null ");
            return;
        }
        // initialize bits allocator
        int timeBits = 64 - 1 - dataCenterIdBits - workerBits - seqBits;
        allocator = new BitsAllocator(timeBits, dataCenterIdBits, workerBits, seqBits);
        // initialize worker id
        workerId = stringRedisTemplate.opsForValue().increment(REDIS_WORK_ID_KEY + dataCenterId, 1) % allocator.getMaxWorkerId();
        Assert.isTrue(workerId <= allocator.getMaxWorkerId(), "WORKER_ID is too big");
        Assert.isTrue(dataCenterId <= allocator.getMaxDataCenterId(), "DATA_CENTER_ID is too big");
        epochSeconds = TimeUnit.MILLISECONDS.toSeconds(DateUtils.parseDate(epochStr, new String[]{DAY_PATTERN}).getTime());
        log.info("Initialized bits dataCenterBits:{}, workerBits:{}, seqBits:{}", dataCenterIdBits, workerBits, seqBits);
        log.info("Initialized nodes, WORKER_ID:{}, DATA_CENTER_ID:{}", workerId, dataCenterId);
        idGenerator = new IDGenerator(dataCenterId, epochSeconds, workerId, allocator);
    }


}
